//! Interpreter module.
//! Contains struct and other items for representing an instance of the wright interpreter.

extern crate regex; 
use self::regex::Regex;
pub mod interpreter_error;
pub mod interactive;
use interpreter::interactive::interactive;
use interpreter::interpreter_error::*;
use errors::Error;
use std::io::{Read, Write};
use std::fs::File;
use std::fmt;
use super::target::Target;

/// Enum for different levels of optimization.
/// See variant documentation for more detail.
#[derive(Debug, Copy, Clone, PartialEq, Eq, Ord, PartialOrd)]
pub enum OptimizationLevel {
    /// No optimizations. Good for debugging. (Default)
    Debug,
    /// Performance optimizations. (Longer compile time, shorter run time)
    Release,
}

/// Intermediate representations that can be emitted during compilations.
#[derive(Debug, Copy, Clone, Eq, PartialEq)]
pub enum Emit {
    /// Tokens generated by scanner.
    /// See [Scanner](../scanner/).
    Tokens,
    /// Lexemes generated during lexical analysis.
    /// See [Lexer](../lexer/).
    Lexemes,
    /// Syntax Tree generated by parser.
    /// See [Parser](../parser/).
    AbstractSyntaxTree
}

impl fmt::Display for Emit {
    fn fmt(&self, f: &mut fmt::Formatter) -> fmt::Result {
        match *self {
            Emit::AbstractSyntaxTree => write!(f, "AST"),
            other => write!(f, "{:?}", other),
        }
    }
}

impl Default for OptimizationLevel {
    fn default() -> Self { OptimizationLevel::Debug }
}

//todo: more testing, better testing
//todo: switch from a file basis to anything that implements std::io::Read, std::io::Write

/// Types of interpreters.
/// See variant level documentation for more details.
#[derive(Debug)]
pub enum Interpreter<'src> {
    /// Interactive (or REPL, as some people call it).
    /// Should be created just using:
    /// ```rust
    /// #use wright::interpreter::Interpreter;
    /// let interactive = Interpreter::Interactive;
    /// ```
    Interactive,
    /// Compiler; takes source file as input and compiles it to the output file.
    /// Should be created using the [`new`] function.
    ///
    /// [`new`]: fn.new
    Compiler {
        emits: Vec<Emit>,
        target: Target,
        output: File,
        optimization: OptimizationLevel,
        file_name: &'src str,
        input: String,
    },
    /// Interpreter; Takes source file, compiles it to a target, and runs it immediately.
    /// Should be created using the [`new`] function.
    ///
    /// [`new`]: fn.new
    Interpreter {
        emits: Vec<Emit>,
        target: Target,
        optimization: OptimizationLevel,
        file_name: &'src str,
        input: String,
    },
    /// Tree walk style interpreter; Takes a source file, parses it, and evaluates it as-is.
    /// Should be created using the [`treewalker`] function.
    ///
    /// [`treewalker`]: fn.treewalker
    TreeWalker {
        emits: Vec<Emit>,
        file_name: &'src str,
        input: String,
    },
}

impl<'source> Interpreter<'source> {
    /// Creates a new tree-walk interpreter, reading the contents of argument file.
    pub fn treewalker(file_name: &'source str, emits: Vec<Emit>) -> Option<Interpreter<'source>> {
        Interpreter::new(file_name, OptimizationLevel::Debug, emits, None, None, true)
    }

    /// Creates a new interpreter, reading the contents of the argument file.
    pub fn new(
        file_name: &'source str,
        level: OptimizationLevel,
        emits: Vec<Emit>,
        target: Option<Target>,
        out: Option<&'source str>,
        run: bool
    ) -> Option<Interpreter<'source>> {
        let mut buf: String = String::new();
        match File::open(file_name) {
            Ok(mut file_handle) => {
                match file_handle.read_to_string(&mut buf) {
                    Ok(_)  => {},
                    Err(_) => {
                        InterpreterError {
                            file_name,
                            reason:   "Could not read input file. (Was it valid UTF-8?)",
                        }.display();
                        return None;
                    }
                };
            },
            Err(_) => {
                InterpreterError {
                    file_name, reason: "Could not open input file. (Does it exist?)",
                }.display();
                return None;
            },
        };
        let mut output_name: String = match out {
            Some(n) => n,
            None    => file_name,
        }.to_string();

        let out_jvm_re = Regex::new(r".+?\.class$").unwrap();
        let out_wasm_re = Regex::new(r".+?\.wasm$").unwrap();
        let out_bf_re = Regex::new(r".+?\.bf$").unwrap();

        match target.unwrap_or(Target::JVM) {
            Target::JVM  => {if !out_jvm_re.is_match(&output_name){output_name.push_str(".class")}},
            Target::WASM => {if !out_wasm_re.is_match(&output_name){output_name.push_str(".wasm")}},
            Target::BrainFuck => {if !out_bf_re.is_match(&output_name){output_name.push_str(".bf")}}
        }
        if !run {
            match File::create(&output_name) {
                Ok(f)  => {
                    Some(Interpreter::Compiler {
                        emits,
                        target: target.unwrap_or(Target::JVM),
                        output: f,
                        optimization: level,
                        file_name,
                        input: buf,
                    })
                },
                Err(_) => {
                    InterpreterError {
                        file_name: &output_name,
                        reason:   "Could not create or open output file.",
                    }.display();
                    None
                },
            }
        } else {
            Some( match target {
                Some(t) => Interpreter::Interpreter {
                    emits,
                    target: t,
                    optimization: level,
                    file_name,
                    input: buf
                },
                None => Interpreter::TreeWalker {emits, file_name, input: buf}
            })
        }

    }
    /// Interpreter execution function
    ///
    /// # Panics:
    /// Always; this function is not yet implemented.
    pub fn run(&self) -> i32 {
        println!("{:?}", self);
        match *self {
            Interpreter::Interactive => interactive(),
            _ => unimplemented!(),
        }
    }
}