# UserFacingError
UserError is an error type that helps you format and print good looking error messages for users of your CLI application. These errors are intended for consumption by a human, not your program. They are divided into 3 parts: summary, reasons and help text.

**Summary:** A String representing a one-line description of your error. A summary is mandatory and is printed boldly in red.

**Reasons:** A vector of Strings explaining in more detail _why_ this error occured. Reasons are optional and if the terminal supports color, the bullet point ('-') will be colored yellow. Each reason will be printed on its own line.

**Help Text:** A String explaining additional information, including what the user can do about the error, or where to file a bug. Help text is optional and if the terminal supports color it will be printed _dimly_.

### Docs
The docs are _really_ good for this crate. In addition to reading this README you can find tons of examples in the documentation generated by Cargo.

# Quickstart
Add the following to your Cargo.toml:
```yaml
[dependencies]
user-error = "1.1.0"
```

Add the following to your main.rs/lib.rs:
```rust
use user_error::UserFacingError;

fn main() {
    let e = UserFacingError::new("Failed to build project") 
                            .reason("Database could not be parsed")
                            .reason("File \"main.db\" not found") 
                            .helptext("Try: touch main.db"); 
    eprintln!("{}", e);
}
```
This prints:
```text
Error: Failed to build project
- Database could not be parsed
- File "main.db" not found
Try: touch main.db
```
If the user has colors enabled on their terminal, it may look something like this:
![Quickstart example of user-error library for Rust](https://xvrqt.sfo2.digitaloceanspaces.com/image-cache/user-error-output.png)

### Print
If for some reason you don't want to follow the format!() convention, you can call print() on a UserError and it will pretty print itself to stderr
```rust
use user_error::UserError;

fn main() {
    let e = UserFacingError::new("Critical Failure!");
    e.print();
}
```
This prints:
```text
Error: Critical Failure!
```

### Print and Exit
Since this error is likely the last thing your program will run you can use this shortcut to print the error and exit the process in an immediate, albeit ungraceful manner. Returns error code 1 to the OS.
```rust
use user_error::UserError;

fn main() {
    let e = UserFacingError::new("Critical Failure!");
    e.print_and_exit();
    eprintln("I am never printed!");
}
```
This prints:
```text
Error: Critical Failure!
```
