// Copyright 2018 Mathew Robinson <chasinglogic@gmail.com>. All rights reserved. Use of this source code is
// governed by the Apache-2.0 license that can be found in the LICENSE file.

use chrono::prelude::*;
use tabwriter::TabWriter;

use std::io;
use std::io::Write;
use std::process::exit;

use taskforge::task::Task;

const DATE_FORMAT: &str = "%Y-%m-%d %I:%M %p";

pub fn print_table(rows: &[Task]) -> Result<(), io::Error> {
    let mut tw = TabWriter::new(io::stdout()).padding(1);

    write!(
        &mut tw,
        "| ID\t| Created Date\t| Completed Date\t| Priority\t| Title\t| Context\t|"
    )?;

    write!(
        &mut tw,
        "\n| --\t| ------------\t| --------------\t| --------\t| -----\t| -------\t|",
    )?;

    for task in rows {
        write!(
            &mut tw,
            "\n| {}\t| {}\t| {}\t| {}\t| {}\t| {}\t|",
            task.id,
            task.created_date.format(DATE_FORMAT),
            task.completed_date
                .map(|dt| format!("{}", dt.format(DATE_FORMAT)))
                .unwrap_or_else(|| "None".to_string()),
            task.priority,
            task.title,
            task.context,
        )?;
    }

    writeln!(&mut tw)?;
    tw.flush().expect("Unable to print to stdout");
    Ok(())
}

pub fn print_json(rows: &[Task]) -> Result<(), io::Error> {
    let mut out = io::stdout();

    match serde_json::to_string_pretty(rows) {
        Ok(s) => writeln!(&mut out, "{}", s)?,
        Err(e) => writeln!(
            &mut out,
            "Unexpected error serializing JSON occurred: {}",
            e
        )?,
    };

    Ok(())
}

// Used to serialize a task as CSV since Notes cannot be serialized nicely.
#[derive(Serialize)]
struct CSVRow {
    id: String,
    title: String,
    context: String,
    body: Option<String>,
    priority: i64,
    created_date: DateTime<Local>,
    completed_date: Option<DateTime<Local>>,
}

impl From<Task> for CSVRow {
    fn from(t: Task) -> CSVRow {
        CSVRow {
            id: t.id,
            title: t.title,
            context: t.context,
            body: t.body,
            priority: t.priority,
            created_date: t.created_date,
            completed_date: t.completed_date,
        }
    }
}

pub fn print_csv(rows: &[Task]) -> Result<(), io::Error> {
    let mut wtr = csv::WriterBuilder::new()
        .has_headers(true)
        .from_writer(io::stdout());

    let csvrows = rows.iter().cloned().map(|t| CSVRow::from(t));
    for row in csvrows {
        if let Err(e) = wtr.serialize(row) {
            println!("Unable to build CSV: {}", e);
            exit(1);
        };
    }

    wtr.flush()
}

pub fn print_text(rows: &[Task]) -> Result<(), io::Error> {
    let mut out = io::stdout();
    for task in rows {
        writeln!(&mut out, "{}: {}", task.id, task.title)?;
    }

    Ok(())
}
