// Copyright 2018 Mathew Robinson <chasinglogic@gmail.com>. All rights reserved. Use of this source code is
// governed by the Apache-2.0 license that can be found in the LICENSE file.

#[macro_use]
extern crate serde_derive;
extern crate docopt;

use docopt::Docopt;
use std::env;
use std::process::Command;

const VERSION: &str = env!("CARGO_PKG_VERSION");
const USAGE: &str = "
Usage: task [options] <command> [<args>...]

A task management CLI that integrates with external services.

Options:
  -h, --help      Print this help message
  --version       Print version and license information
  -v, --verbose   Print debug information, useful when submitting bug reports!

Commands:
   help                Print usage information about task commands
   add (new, a)        Add a new task to the list
   next (n)            Print the next or \"current\" task in the list
   todo                Print incomplete tasks in the list
   edit (e)            Edit task data as a toml file
   complete (done, d)  Complete tasks in the list.
   query (q, s, list)  Search or list tasks in the list
   workon              Move a task to the top of the list

See 'task help <command>' for more information on a specific command.
";

#[derive(Deserialize, Debug)]
struct Args {
    flag_verbose: bool,
    flag_version: bool,
    flag_help: bool,
    arg_command: String,
    arg_args: Vec<String>,
}

#[inline]
fn alias(cmd: &str) -> &str {
    match cmd {
        "new" | "a" => "add",
        "n" | "current" => "next",
        "e" => "edit",
        "done" | "d" => "complete",
        "q" | "list" | "s" | "search" => "query",
        x => x,
    }
}

fn main() {
    let args: Args = Docopt::new(USAGE)
        .and_then(|d| d.options_first(true).help(false).deserialize())
        .unwrap_or_else(|e| e.exit());

    if args.flag_version {
        println!("task version {}", VERSION);
    } else if args.arg_command == "" && args.flag_help {
        println!("{}", USAGE);
    } else {
        let whereami = env::current_exe().unwrap();
        let subcommand = alias(&args.arg_command);
        let subcommand_path = format!("{}-{}", whereami.to_str().unwrap(), subcommand);
        match Command::new(subcommand_path).args(&args.arg_args).spawn() {
            Ok(mut child) => {
                // TODO: remove this expect.
                child.wait().expect("problem running subcommand.");
            }
            Err(e) => {
                if args.flag_verbose {
                    println!("{}", e);
                }

                println!("{}: is not a known subcommand", args.arg_command);
            }
        };
    }
}

#[cfg(test)]
pub mod test {
    use super::alias;

    macro_rules! test_aliases {
        ($($alias:expr, $command:expr,)*) => {
            #[test]
            fn test_aliases() {
                $(
                    assert_eq!(alias($alias), $command);
                )*
            }
        }

    }

    test_aliases! {
        "a", "add",
        "new", "add",

        "n", "next",
        "e", "edit",

        "q", "query",
        "s", "query",
        "list", "query",
        "search", "query",

        "d", "complete",
        "done", "complete",

        "nope", "nope",
    }
}
