// Copyright 2018 Mathew Robinson <chasinglogic@gmail.com>. All rights reserved. Use of this source code is
// governed by the Apache-2.0 license that can be found in the LICENSE file.

#[macro_use]
extern crate serde_derive;

extern crate docopt;
extern crate task_cli;
extern crate taskforge;

use docopt::Docopt;

use std::process::exit;

use task_cli::config::Config;
use task_cli::printing::*;
use taskforge::query::ast::{Expression, AST};
use taskforge::query::token::Operator;

const USAGE: &str = "
Usage: task-todo [options]

A convenience command for listing tasks which are incomplete.

Options:
    -o <format>, --output <format>  How to display the tasks which match the
                                    query. Available formats are: json, csv,
                                    table, text. See 'task list --help' for
                                    more information on how each format is
                                    displayed. [default: table]
";

#[derive(Deserialize, Debug)]
struct Args {
    flag_output: String,
}

fn main() {
    let cfg = Config::load();
    let mut list = cfg.list();
    let args: Args = Docopt::new(USAGE)
        .and_then(|d| d.deserialize())
        .unwrap_or_else(|e| e.exit());

    let ast = AST {
        expression: Expression::Infix(
            Box::new(Expression::String("completed".to_string())),
            Operator::EQ,
            Box::new(Expression::Bool(false)),
        ),
    };

    let res = match list.search(ast) {
        Ok(tasks) => match args.flag_output.as_ref() {
            "table" => print_table(&tasks),
            "json" => print_json(&tasks),
            "text" => print_text(&tasks),
            "csv" => print_csv(&tasks),
            _ => print_table(&tasks),
        },
        Err(e) => Err(e.into_io_err()),
    };

    if res.is_err() {
        println!("Unable to query tasks: {}", res.unwrap_err());
        exit(1)
    }
}
