// Copyright 2018 Mathew Robinson <chasinglogic@gmail.com>. All rights reserved. Use of this source code is
// governed by the Apache-2.0 license that can be found in the LICENSE file.

#[macro_use]
extern crate serde_derive;

extern crate csv;
extern crate docopt;
extern crate task_cli;
extern crate taskforge;

use docopt::Docopt;

use std::process::exit;

use task_cli::config::Config;
use task_cli::printing::*;
use taskforge::query::ast::AST;
use taskforge::query::Parser;

const USAGE: &str = "
Usage: task-query [options] [<query>...]

Search or list tasks in this list.

QUERY will be concatenated using spaces and will be interpreted using the
Taskforge Query Language.

If no query is provided all tasks will be returned.

You can view information about the Taskforge Query Language using 'task help ql'
or by visiting:

http://taskforge.io/docs/query_language

Options:
    -o <format>, --output <format>  How to display the tasks which match the
                                    query. Available formats are: json, csv,
                                    table, text which are described below.
                                    [default: table]

output formats:

   Text output format is the same as for task next where each task will be
   printed on one line with the format:

   $TASK_ID: $TASK_TITLE

   Table output format lists tasks in an ascii table and it looks like this:

   | ID  | Created Date  | Completed Date  | Priority  | Title  | Context  |
   | --- | ------------- | --------------- | --------- | ------ | -------- |
   | $ID | $CREATED_DATE | $COMPLETED_DATE | $PRIORITY | $TITLE | $CONTEXT |

   JSON output format will \"pretty print\" a JSON array of the tasks in the list
   to stdout.  They will be properly indented 4 spaces and should be fairly
   human readable.  This is useful for migrating from one list implementation
   for another as you can redirect this output to a file then import it with:
   'task add --from-file $YOUR_JSON_FILE'

   CSV will output all task metadata in a csv format. It will write to stdout
   so you can use shell redirection to put it into a csv file like so:

   'task list --output csv > my_tasks.csv'

   This is useful for importing tasks into spreadsheet programs like Excel.
";

#[derive(Deserialize, Debug)]
struct Args {
    arg_query: Vec<String>,
    flag_output: String,
}

fn main() {
    let cfg = Config::load();
    let mut list = cfg.list();
    let args: Args = Docopt::new(USAGE)
        .and_then(|d| d.deserialize())
        .unwrap_or_else(|e| e.exit());

    let ast = if args.arg_query.is_empty() {
        AST::empty()
    } else {
        let query = args.arg_query.join(" ");
        match Parser::from(query.as_ref()).parse() {
            Ok(st) => st,
            Err(e) => {
                println!("Problem parsing query: {}", e);
                exit(1);
            }
        }
    };

    let res = match list.search(ast) {
        Ok(tasks) => match args.flag_output.as_ref() {
            "json" => print_json(&tasks),
            "text" => print_text(&tasks),
            "csv" => print_csv(&tasks),
            "table" => print_table(&tasks),
            _ => print_table(&tasks),
        },
        Err(e) => Err(e.into_io_err()),
    };

    if res.is_err() {
        println!("Unable to query tasks: {}", res.unwrap_err());
        exit(1)
    }
}
