// Copyright 2018 Mathew Robinson <chasinglogic@gmail.com>. All rights reserved. Use of this source code is
// governed by the Apache-2.0 license that can be found in the LICENSE file.

#[macro_use]
extern crate serde_derive;

extern crate docopt;
extern crate task_cli;
extern crate taskforge;

use docopt::Docopt;

use std::process::exit;
use task_cli::config::Config;
use taskforge::list::Error;

const USAGE: &str = "
Usage: task-next [options]

Print the \"next\" or \"current\" task. This is calculated by the list as the
highest priority, oldest task in the list.

Default output format is:

$TASK_ID: $TASK_TITLE

You can modify the output with the options below.

Options:
    -h, --help        Print this usage information
    -i, --id-only     Print only the task ID
    -t, --title-only  Print only the task title
";

#[derive(Deserialize, Debug)]
struct Args {
    flag_id_only: bool,
    flag_title_only: bool,
}

fn main() {
    let cfg = Config::load();
    let mut list = cfg.list();
    let args: Args = Docopt::new(USAGE)
        .and_then(|d| d.deserialize())
        .unwrap_or_else(|e| e.exit());

    match list.current() {
        Ok(task) => {
            if args.flag_id_only {
                println!("{}", task.id);
            } else if args.flag_title_only {
                println!("{}", task.title);
            } else {
                println!("{}: {}", task.id, task.title)
            }
        }
        Err(e) => match e {
            Error::NotFound => println!("All done!"),
            err => {
                println!("Unable to retrieve current task: {}", err);
                exit(1);
            }
        },
    }
}
