// Copyright 2018 Mathew Robinson <chasinglogic@gmail.com>. All rights reserved. Use of this source code is
// governed by the Apache-2.0 license that can be found in the LICENSE file.

#[macro_use]
extern crate serde_derive;
extern crate docopt;
extern crate serde_json;
extern crate task_cli;
extern crate taskforge;

use std::fs::File;
use std::io;
use std::io::Read;
use std::path::PathBuf;
use std::process::exit;

use docopt::Docopt;

use task_cli::config::Config;
use taskforge::list::Error;
use taskforge::task::Task;

const USAGE: &str = "
Usage: task-add [options] [<title>...]

Add or import tasks into the list.

Options:
   -h, --help                            Print this usage information
   -p <priority>, --priority <priority>  Create the task with the indicated
                                         priority, this can be an integer or
                                         float [default: 1]
   -b <body>, --body <body>              The body or \"description\" of the task
   -c <context>, --context <context>     The context in which to create the task [default: default]
   -t, --top                             Make this task the top priority

Import Options:
   -f <file>, --from-file <file>  Import tasks from the indicated JSON file

If an import option is provided all other options are ignored.
";

#[derive(Deserialize, Debug)]
struct Args {
    arg_title: Vec<String>,
    flag_from_file: Option<PathBuf>,
    flag_priority: i64,
    flag_body: String,
    flag_context: String,
    flag_top: bool,
}

fn main() {
    let args: Args = Docopt::new(USAGE)
        .and_then(|d| d.deserialize())
        .unwrap_or_else(|e| e.exit());

    if let Some(filename) = args.flag_from_file {
        match import_from_file(filename) {
            Ok(()) => exit(0),
            Err(e) => {
                println!("Problem importing from file: {}", e);
                exit(1);
            }
        }
    }

    let cfg = Config::load();
    let mut list = cfg.list();

    let mut task = Task::new(&args.arg_title.join(" "))
        .with_priority(args.flag_priority)
        .with_body(&args.flag_body)
        .with_context(&args.flag_context);

    if args.flag_top {
        let highest_priority = match list.current() {
            Ok(task) => task.priority,
            Err(e) => match e {
                Error::NotFound => 1,
                err => {
                    println!("Unable to retrieve current task: {}", err);
                    exit(1);
                }
            },
        };

        task.priority = highest_priority + 1;
    }

    match list.add(task) {
        Ok(_) => (),
        Err(e) => {
            println!("Error adding task: {}", e);
            exit(1);
        }
    }
}

fn import_from_file(filename: PathBuf) -> Result<(), io::Error> {
    let cfg = Config::load();
    let mut list = cfg.list();

    let mut file = File::open(filename)?;
    let mut data = String::new();
    file.read_to_string(&mut data)?;

    let tasks: Vec<Task> = match serde_json::from_str(&data) {
        Ok(tasks) => tasks,
        Err(e) => {
            println!("Unable to parse data: {}", e);
            exit(1);
        }
    };

    for task in tasks.iter() {
        match list.add(task.clone()) {
            Ok(_) => (),
            Err(e) => {
                println!("Error adding task: {}", e);
                exit(1);
            }
        };
    }

    Ok(())
}
