// Copyright 2018 Mathew Robinson <chasinglogic@gmail.com>. All rights reserved. Use of this source code is
// governed by the Apache-2.0 license that can be found in the LICENSE file.


#[macro_use]
extern crate serde_derive;

extern crate docopt;
extern crate task_cli;
extern crate taskforge;

use docopt::Docopt;

use std::process::exit;
use task_cli::config::Config;
use taskforge::list::Error;

const USAGE: &str = "
Usage: task-workon <id>

Find task with id and make it so the priority of the task is 0.1 higher than
that of the current highest priority task. Effectively making it the \"current\"
task in Taskforge terms.
";

#[derive(Deserialize, Debug)]
struct Args {
    arg_id: String,
}

fn main() {
    let cfg = Config::load();
    let mut list = cfg.list();
    let args: Args = Docopt::new(USAGE)
        .and_then(|d| d.deserialize())
        .unwrap_or_else(|e| e.exit());

    let highest_priority = match list.current() {
        Ok(task) => task.priority,
        Err(e) => match e {
            Error::NotFound => 1.0,
            err => {
                println!("Unable to retrieve current task: {}", err);
                exit(1);
            }
        },
    };

    let mut task = match list.find_by_id(args.arg_id.clone()) {
        Ok(task) => task,
        Err(e) => match e {
            Error::NotFound => {
                println!("No task with that id exists.");
                exit(1);
            }
            err => {
                println!("Unable to retrieve task with id {}: {}", args.arg_id, err);
                exit(1);
            }
        },
    };

    task.priority = highest_priority + 0.1;
    let task_id = task.id.clone();
    match list.update(task) {
        Ok(_) => (),
        Err(e) => {
            println!("Unable to update task {}: {}", task_id, e);
            exit(1);
        }
    }
}
