// Copyright 2018 Mathew Robinson <chasinglogic@gmail.com>. All rights reserved. Use of this source code is
// governed by the Apache-2.0 license that can be found in the LICENSE file.


#[macro_use]
extern crate serde_derive;

extern crate docopt;
extern crate task_cli;
extern crate taskforge;
extern crate tempfile;
extern crate toml;

use docopt::Docopt;

use std::env::var_os;
use std::fs::File;
use std::io::{Read, Write};
use std::process::exit;
use std::process::Command;

use task_cli::config::Config;
use taskforge::list::Error;

const USAGE: &str = "
Usage: task-edit <id>

Edit the task indicated by ID as a toml file. If no ID given opens the current
task.

Will use $EDITOR if set and if not will attempt to find an editor based on
platform.
";

#[derive(Deserialize, Debug)]
struct Args {
    arg_id: String,
}

fn editor_program() -> String {
    match var_os("EDITOR") {
        Some(editor) => editor.into_string().unwrap(),
        None => if cfg!(windows) {
            "notepad.exe".to_string()
        } else {
            "vi".to_string()
        },
    }
}

fn main() {
    let cfg = Config::load();
    let mut list = cfg.list();
    let args: Args = Docopt::new(USAGE)
        .and_then(|d| d.deserialize())
        .unwrap_or_else(|e| e.exit());

    let mut task = match list.find_by_id(args.arg_id) {
        Ok(task) => task,
        Err(e) => match e {
            Error::NotFound => {
                println!("No task with that id exists.");
                exit(1);
            }
            err => {
                println!("Unable to retrieve task: {}", err);
                exit(1);
            }
        },
    };

    let tmp = tempfile::tempdir().expect("Unable to create a temporary directory");
    let file_path = tmp.path().join("task.toml");
    let mut toml_str = toml::to_string(&task).expect("Unable to serialize task");

    {
        let mut file = File::create(file_path.clone()).expect("Unable to create temporary file");
        file.write_all(toml_str.as_bytes())
            .expect("Unable to write to temporary file");
        file.sync_all().expect("Unable to persist temporary file");
    }

    let editor_cmd_str = editor_program();
    let mut iter = editor_cmd_str.split_whitespace();
    let editor_exe = iter.next().unwrap();
    let mut editor_args: Vec<&str> = iter.collect();
    let file_str = file_path.to_str().unwrap();
    editor_args.push(file_str);

    if let Err(e) = Command::new(editor_exe)
        .args(&editor_args)
        .spawn()
        .unwrap()
        .wait()
    {
        println!("Problem opening temporary file: {}", e);
    }

    toml_str = "".to_string();

    let mut reopened_file = File::open(file_path.clone()).expect("Unable to reopen temporary file");
    reopened_file
        .read_to_string(&mut toml_str)
        .expect("Unable to read temporary file");

    task = match toml::from_str(&toml_str) {
        Ok(t) => t,
        Err(e) => {
            println!("Unable to parse toml: {}", e);
            exit(1);
        }
    };

    match list.update(task) {
        Ok(_) => (),
        Err(e) => {
            println!("Unable to update task: {}", e);
            exit(1);
        }
    }
}
