// Copyright 2018 Mathew Robinson <chasinglogic@gmail.com>. All rights reserved. Use of this source code is
// governed by the Apache-2.0 license that can be found in the LICENSE file.


#[macro_use]
extern crate serde_derive;

extern crate docopt;
extern crate task_cli;
extern crate taskforge;

use docopt::Docopt;

use std::process::exit;
use task_cli::config::Config;
use taskforge::list::Error;

const USAGE: &str = "
Usage: task-complete [<id>...]

Complete tasks by id. If no id's are provided then the current task indicated by
'task next' is completed.
";

#[derive(Deserialize, Debug)]
struct Args {
    arg_id: Vec<String>,
}

fn main() {
    let cfg = Config::load();
    let mut list = cfg.list();
    let args: Args = Docopt::new(USAGE)
        .and_then(|d| d.deserialize())
        .unwrap_or_else(|e| e.exit());

    let mut ids = args.arg_id;
    if ids.is_empty() {
        match list.current() {
            Ok(task) => ids = vec![task.id],
            Err(e) => match e {
                Error::NotFound => {
                    println!("No current task found and no id provided.");
                    exit(0);
                }
                err => {
                    println!("Unable to retrieve current task: {}", err);
                    exit(1);
                }
            },
        }
    }

    for id in ids.iter() {
        if let Err(e) = list.complete(id.to_string()) {
            println!("Unable to complete task {}: {}", id, e);
            exit(1);
        }
    }
}
