use webcore::value::{Reference, Value};
use webcore::try_from::{TryFrom, TryInto};
use webapi::event_target::EventTarget;
use webapi::window::Window;

/// The `IEvent` interface represents any event which takes place in the DOM; some
/// are user-generated (such as mouse or keyboard events), while others are
/// generated by APIs (such as events that indicate an animation has finished
/// running, a video has been paused, and so forth). There are many types of event,
/// some of which use other interfaces based on the main `IEvent` interface. `IEvent`
/// itself contains the properties and methods which are common to all events.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event)
pub trait IEvent: AsRef< Reference > + TryFrom< Value > {
    /// Indicates whether this event bubbles upward through the DOM.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event)
    #[inline]
    fn bubbles( &self ) -> bool {
        js!(
            return @{self.as_ref()}.bubbles;
        ).try_into().unwrap()
    }

    /// A historical alias to `Event.stopPropagation()`.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event/cancelBubble)
    #[inline]
    fn cancel_bubble( &self ) -> bool {
        js!(
            return @{self.as_ref()}.cancelBubble;
        ).try_into().unwrap()
    }

    /// A historical alias to `Event.stopPropagation()`.
    /// Setting this to `true` before returning from an event handler will stop propagation
    /// of the event.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event/cancelBubble)
    #[inline]
    fn set_cancel_bubble( &self, value: bool ) {
        js! { @(no_return)
            @{self.as_ref()}.cancelBubble = @{value};
        }
    }

    /// Indicates whether the event is cancelable.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event/cancelable)
    #[inline]
    fn cancelable( &self ) -> bool {
        js!(
            return @{self.as_ref()}.cancelable;
        ).try_into().unwrap()
    }

    /// A reference to the currently registered target of this event.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event/currentTarget)
    #[inline]
    fn current_target( &self ) -> Option< EventTarget > {
        js!(
            return @{self.as_ref()}.currentTarget;
        ).try_into().ok()
    }

    /// Indicates whether `preventDefault` has been called on this event.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event/defaultPrevented)
    #[inline]
    fn default_prevented( &self ) -> bool {
        js!(
            return @{self.as_ref()}.defaultPrevented;
        ).try_into().unwrap()
    }

    /// Indicates which phase of event flow is currently being evaluated.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event/eventPhase)
    fn event_phase( &self ) -> EventPhase {
        match js!(
            return @{self.as_ref()}.eventPhase;
        ).try_into().unwrap() {
            0 => EventPhase::None,
            1 => EventPhase::Capturing,
            2 => EventPhase::AtTarget,
            3 => EventPhase::Bubbling,
            _ => unreachable!("Unexpected EventPhase type"),
        }
    }

    /// Prevents any further listeners from being called for this event.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event/stopImmediatePropagation)
    #[inline]
    fn stop_immediate_propagation( &self ) {
        js! { @(no_return)
            @{self.as_ref()}.stopImmediatePropagation();
        }
    }

    /// Stops the propagation of this event to descendants in the DOM.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event/stopPropagation)
    #[inline]
    fn stop_propagation( &self ) {
        js! { @(no_return)
            @{self.as_ref()}.stopPropagation();
        }
    }


    /// Returns a reference to the target to which this event was originally registered.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event/target)
    #[inline]
    fn target( &self ) -> Option< EventTarget > {
        js!(
            return @{self.as_ref()}.target;
        ).try_into().ok()
    }

    /// Returns the time in milliseconds at which this event was created.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event/timeStamp)
    #[inline]
    fn time_stamp( &self ) -> Option< f64 > {
        js!(
            return @{self.as_ref()}.timeStamp;
        ).try_into().ok()
    }

    /// Indicates whether the event was generated by a user action.
    #[inline]
    fn is_trusted( &self ) -> bool {
        js!(
            return @{self.as_ref()}.isTrusted;
        ).try_into().unwrap()
    }

    /// Returns a string containing the type of event. It is set when
    /// the event is constructed and is the name commonly used to refer
    /// to the specific event.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event/type)
    #[inline]
    fn event_type( &self ) -> String {
        js!(
            return @{self.as_ref()}.type;
        ).try_into().unwrap()
    }

    /// Cancels the event if it is cancelable, without
    /// stopping further propagation of the event.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
    #[inline]
    fn prevent_default( &self ) {
        js! { @(no_return)
            @{self.as_ref()}.preventDefault();
        }
    }
}

/// Indicates the phase of event flow during event proessing.
#[derive(Clone, Copy, Debug, Eq, PartialEq)]
pub enum EventPhase {
    /// No event is currently being processed.
    None,
    /// The event is being propagated down through the target's ancestors.
    Capturing,
    /// The target is currently processing the event.
    AtTarget,
    /// The event is propagating back up through the target's ancestors.
    Bubbling,
}

/// A trait representing a concrete event type.
pub trait ConcreteEvent: IEvent {
    /// A string representing the event type.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event/type)
    const EVENT_TYPE: &'static str;
}

/// A reference to a JavaScript object which implements the [IEvent](trait.IEvent.html)
/// interface.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event)
pub struct Event( Reference );

impl IEvent for Event {}

reference_boilerplate! {
    Event,
    instanceof Event
}

/// The `ChangeEvent` is fired for input, select, and textarea
/// elements when a change to the element's value is committed
/// by the user. Unlike the input event, the change event is not
/// necessarily fired for each change to an element's value.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/change)
pub struct ChangeEvent( Reference );

impl IEvent for ChangeEvent {}
impl ConcreteEvent for ChangeEvent {
    const EVENT_TYPE: &'static str = "change";
}

reference_boilerplate! {
    ChangeEvent,
    instanceof Event
    convertible to Event
}

/// The `IUiEvent` interface represents simple user interface events.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/UIEvent)
pub trait IUiEvent: IEvent {
    /// Provides the current click count for this event, if applicable.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/UIEvent/detail)
    #[inline]
    fn detail( &self ) -> i32 {
        js!(
            return @{self.as_ref()}.detail;
        ).try_into().unwrap()
    }

    /// Returns the `WindowProxy` that generated the event.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/UIEvent/view)
    #[inline]
    fn view( &self ) -> Option< Window > {
        js!(
            return @{self.as_ref()}.view;
        ).try_into().ok()
    }
}

/// A reference to a JavaScript object which implements the [IUiEvent](trait.IUiEvent.html)
/// interface.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/UIEvent)
pub struct UiEvent( Reference );

impl IEvent for UiEvent {}
impl IUiEvent for UiEvent {}

reference_boilerplate! {
    UiEvent,
    instanceof UIEvent
    convertible to Event
}

/// The `LoadEvent` is fired when a resource and its dependent resources have finished loading.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/load)
pub struct LoadEvent( Reference );

impl IEvent for LoadEvent {}
impl IUiEvent for LoadEvent {}
impl ConcreteEvent for LoadEvent {
    const EVENT_TYPE: &'static str = "load";
}

reference_boilerplate! {
    LoadEvent,
    instanceof UIEvent
    convertible to Event
    convertible to UiEvent
}

/// The `IMouseEvent` interface represents events that occur due to the user
/// interacting with a pointing device (such as a mouse).
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent)
pub trait IMouseEvent: IUiEvent {
    /// Returns whether the Alt key was down when this event was fired.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/altKey)
    #[inline]
    fn alt_key( &self ) -> bool {
        js!(
            return @{self.as_ref()}.altKey;
        ).try_into().unwrap()
    }

    /// Indicates the mouse button that fired this event.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/button)
    fn button( &self ) -> MouseButton {
        match js!(
            return @{self.as_ref()}.button;
        ).try_into().unwrap() {
            0 => MouseButton::Left,
            1 => MouseButton::Wheel,
            2 => MouseButton::Right,
            3 => MouseButton::Button4,
            4 => MouseButton::Button5,
            _ => unreachable!("Unexpected MouseEvent.button value"),
        }
    }

    /// Indicates which mouse buttons were down when this event was fired.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/buttons)
    fn buttons( &self ) -> MouseButtonsState {
        MouseButtonsState(
            js!(
                return @{self.as_ref()}.buttons;
            ).try_into().unwrap()
        )
    }

    /// Returns the X position in the application's client area where this event occured.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/clientX)
    #[inline]
    fn client_x( &self ) -> f64 {
        js!(
            return @{self.as_ref()}.clientX;
        ).try_into().unwrap()
    }

    /// Returns the Y position in the application's client area where this event occured.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/clientY)
    #[inline]
    fn client_y( &self ) -> f64 {
        js!(
            return @{self.as_ref()}.clientY;
        ).try_into().unwrap()
    }

    /// Indicates whether the Ctrl key was down when this event fired.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/ctrlKey)
    #[inline]
    fn ctrl_key( &self ) -> bool {
        js!(
            return @{self.as_ref()}.ctrlKey;
        ).try_into().unwrap()
    }

    /// Returns the current state of the specified modifier key.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/getModifierState)
    #[inline]
    fn get_modifier_state( &self, key: ModifierKey ) -> bool {
        get_event_modifier_state( self, key )
    }

    /// Indicates whether the Meta key was down when this event fired.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/metaKey)
    #[inline]
    fn meta_key( &self ) -> bool {
        js!(
            return @{self.as_ref()}.metaKey;
        ).try_into().unwrap()
    }

    /// Returns the change in X coordinate of the pointer between this event and the previous
    /// MouseMove event.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/movementX)
    #[inline]
    fn movement_x( &self ) -> f64 {
        js!(
            return @{self.as_ref()}.movementX;
        ).try_into().unwrap()
    }

    /// Returns the change in Y coordinate of the pointer between this event and the previous
    /// MouseMove event.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/movementX)
    #[inline]
    fn movement_y( &self ) -> f64 {
        js!(
            return @{self.as_ref()}.movementY;
        ).try_into().unwrap()
    }

    /// Returns the ID of the hit region affected by the event.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/region)
    #[inline]
    fn region( &self ) -> Option< String > {
        js!(
            return @{self.as_ref()}.region;
        ).try_into().ok()
    }

    /// Returns the secondary target of this event, if any.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/relatedTarget)
    #[inline]
    fn related_target( &self ) -> Option< EventTarget > {
        js!(
            return @{self.as_ref()}.relatedTarget;
        ).try_into().ok()
    }

     /// Returns the X position of the pointer in screen coordinates.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/screenX)
    #[inline]
    fn screen_x( &self ) -> f64 {
        js!(
            return @{self.as_ref()}.screenX;
        ).try_into().unwrap()
    }

    /// Returns the Y position of the pointer in screen coordinates.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/screenY)
    #[inline]
    fn screen_y( &self ) -> f64 {
        js!(
            return @{self.as_ref()}.screenY;
        ).try_into().unwrap()
    }

    /// Indicates whether the Shift key was down when this event was fired.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent/shiftKey)
    #[inline]
    fn shift_key( &self ) -> bool {
        js!(
            return @{self.as_ref()}.shiftKey;
        ).try_into().unwrap()
    }
}

/// Represents buttons on a mouse during mouse events.
#[derive(Clone, Copy, Debug, PartialEq, Eq)]
pub enum MouseButton {
    /// The left mouse button.
    Left,
    /// The mouse wheel/middle mouse button.
    Wheel,
    /// The right mouse button.
    Right,
    /// The fourth mouse button (browser back).
    Button4,
    /// The fifth mouse button (browser forward).
    Button5,
}

/// Represents the state of mouse buttons in a `MouseEvent`.
#[derive(Copy, Clone, Debug, PartialEq, Eq)]
pub struct MouseButtonsState(u8);

impl MouseButtonsState {
    pub fn is_down(&self, button: MouseButton) -> bool {
        match button {
            MouseButton::Left => self.0 & 0b1 != 0,
            MouseButton::Right => self.0 & 0b10 != 0,
            MouseButton::Wheel => self.0 & 0b100 != 0,
            MouseButton::Button4 => self.0 & 0b1000 != 0,
            MouseButton::Button5 => self.0 & 0b1_0000 != 0,
        }
    }
}

/// A reference to a JavaScript object which implements the [IMouseEvent](trait.IMouseEvent.html)
/// interface.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent)
pub struct MouseEvent( Reference );

impl IEvent for MouseEvent {}
impl IUiEvent for MouseEvent {}
impl IMouseEvent for MouseEvent {}

reference_boilerplate! {
    MouseEvent,
    instanceof MouseEvent
    convertible to Event
    convertible to UiEvent
}

/// The `ClickEvent` is fired when a pointing device button (usually a
/// mouse's primary button) is pressed and released on a single element.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/click)
pub struct ClickEvent( Reference );

impl IEvent for ClickEvent {}
impl IUiEvent for ClickEvent {}
impl IMouseEvent for ClickEvent {}
impl ConcreteEvent for ClickEvent {
    const EVENT_TYPE: &'static str = "click";
}

reference_boilerplate! {
    ClickEvent,
    instanceof MouseEvent
    convertible to Event
    convertible to UiEvent
    convertible to MouseEvent
}

/// The `DoubleClickEvent` is fired when a pointing device button
/// (usually a mouse's primary button) is clicked twice on a single
/// element.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/dblclick)
pub struct DoubleClickEvent( Reference );

impl IEvent for DoubleClickEvent {}
impl IUiEvent for DoubleClickEvent {}
impl IMouseEvent for DoubleClickEvent {}
impl ConcreteEvent for DoubleClickEvent {
    const EVENT_TYPE: &'static str = "dblclick";
}

reference_boilerplate! {
    DoubleClickEvent,
    instanceof MouseEvent
    convertible to Event
    convertible to UiEvent
    convertible to MouseEvent
}

/// `IKeyboardEvent` objects describe a user interaction with the
/// keyboard. Each event describes a key; the event type identifies
/// what kind of activity was performed.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent)
pub trait IKeyboardEvent: IEvent {
    /// Indicates whether the Alt key was down when this event was fired.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/altKey)
    #[inline]
    fn alt_key( &self ) -> bool {
        js!(
            return @{self.as_ref()}.altKey;
        ).try_into().unwrap()
    }

    /// Returns a code value that indicates the physical key pressed on the keyboard.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/code)
    #[inline]
    fn code( &self ) -> String {
        js!(
            return @{self.as_ref()}.code;
        ).try_into().unwrap()
    }

    /// Returns whether the Ctrl key was down when this event was fired.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/ctrlKey)
    #[inline]
    fn ctrl_key( &self ) -> bool {
        js!(
            return @{self.as_ref()}.ctrlKey;
        ).try_into().unwrap()
    }


    /// Returns whether a modifier key was down when this event was fired.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/getModifierState)
    #[inline]
    fn get_modifier_state( &self, key: ModifierKey ) -> bool {
        get_event_modifier_state( self, key )
    }

    /// Returns whether this event was fired during composition.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/isComposing)
    #[inline]
    fn is_composing( &self ) -> bool {
        js!(
            return @{self.as_ref()}.isComposing;
        ).try_into().unwrap()
    }

    /// Returns the location of the key on the keyboard.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/location)
    fn location( &self ) -> KeyboardLocation {
        match js!(
            return @{self.as_ref()}.location;
        ).try_into().unwrap() {
            0 => KeyboardLocation::Standard,
            1 => KeyboardLocation::Left,
            2 => KeyboardLocation::Right,
            3 => KeyboardLocation::Numpad,
            4 => KeyboardLocation::Mobile,
            5 => KeyboardLocation::Joystick,
            _ => unreachable!("Unexpected KeyboardEvent.location value"),
        }
    }

    /// Returns the value of a key or keys pressed by the user.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key)
    #[inline]
    fn key( &self ) -> String {
        js!(
            return @{self.as_ref()}.key;
        ).into_string().unwrap()
    }

    /// Indicates whether the Meta key was down when this event was fired.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/metaKey)
    #[inline]
    fn meta_key( &self ) -> bool {
        js!(
            return @{self.as_ref()}.metaKey;
        ).try_into().unwrap()
    }

    /// Indicates whether the key is held down such that it is repeating.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/repeat)
    #[inline]
    fn repeat( &self ) -> bool {
        js!(
            return @{self.as_ref()}.repeat;
        ).try_into().unwrap()
    }

    /// Indicates whether the Shift key was down when this event was fired.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/shiftKey)
    #[inline]
    fn shift_key( &self ) -> bool {
        js!(
            return @{self.as_ref()}.shiftKey;
        ).try_into().unwrap()
    }
}

/// A modifier key on the keyboard.
#[allow(missing_docs)]
#[derive(Clone, Copy, Debug, Eq, PartialEq)]
pub enum ModifierKey {
    Alt,
    AltGr,
    CapsLock,
    Ctrl,
    Function,
    FunctionLock,
    Hyper,
    Meta,
    NumLock,
    OS,
    ScrollLock,
    Shift,
    Super,
    Symbol,
    SymbolLock,
}

/// Used by KeyboardEvent and MouseEvent to get the state of a modifier key.
fn get_event_modifier_state< T: IEvent >( event: &T, key: ModifierKey ) -> bool {
    js!(
        return @{event.as_ref()}.getModifierState( @{
            match key {
                ModifierKey::Alt => "Alt",
                ModifierKey::AltGr => "AltGraph",
                ModifierKey::CapsLock => "CapsLock",
                ModifierKey::Ctrl => "Control",
                ModifierKey::Function => "Fn",
                ModifierKey::FunctionLock => "FnLock",
                ModifierKey::Hyper => "Hyper",
                ModifierKey::Meta => "Meta",
                ModifierKey::NumLock => "NumLock",
                ModifierKey::OS => "OS",
                ModifierKey::ScrollLock => "ScrollLock",
                ModifierKey::Shift => "Shift",
                ModifierKey::Super => "Super",
                ModifierKey::Symbol => "Symbol",
                ModifierKey::SymbolLock => "SymbolLock",
            }
        } );
    ).try_into().unwrap()
}

/// The location on the keyboard of a key.
#[derive(Clone, Copy, Debug, Eq, PartialEq)]
pub enum KeyboardLocation {
    /// The key has only one version, or the location can't be distinguished.
    Standard,
    /// The left-hand version of a key.
    Left,
    /// The right-hand version of a key.
    Right,
    /// The key was on a numeric pad.
    Numpad,
    /// The key was on a mobile device.
    Mobile,
    /// The key was on a joystick.
    Joystick,
}

/// A reference to a JavaScript object which implements the [IKeyboardEvent](trait.IKeyboardEvent.html)
/// interface.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent)
pub struct KeyboardEvent( Reference );

impl IEvent for KeyboardEvent {}
impl IKeyboardEvent for KeyboardEvent {}

reference_boilerplate! {
    KeyboardEvent,
    instanceof KeyboardEvent
    convertible to Event
}

/// The `KeypressEvent` is fired when a key is pressed down, and that
/// key normally produces a character value.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/keypress)
pub struct KeypressEvent( Reference );

impl IEvent for KeypressEvent {}
impl IKeyboardEvent for KeypressEvent {}
impl ConcreteEvent for KeypressEvent {
    const EVENT_TYPE: &'static str = "keypress";
}

reference_boilerplate! {
    KeypressEvent,
    instanceof KeyboardEvent
    convertible to Event
    convertible to KeyboardEvent
}

/// The `IFocusEvent` interface represents focus-related
/// events.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/FocusEvent)
pub trait IFocusEvent: IEvent {
    /// Returns the secondary target of this event, if any.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/FocusEvent/relatedTarget)
    #[inline]
    fn related_target( &self ) -> Option< EventTarget > {
        js!(
            return @{self.as_ref()}.relatedTarget;
        ).try_into().ok()
    }
}

/// A reference to a JavaScript object which implements the [IFocusEvent](trait.IFocusEvent.html)
/// interface.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/FocusEvent)
pub struct FocusRelatedEvent( Reference );

impl IEvent for FocusRelatedEvent {}
impl IFocusEvent for FocusRelatedEvent {}

reference_boilerplate! {
    FocusRelatedEvent,
    instanceof FocusEvent
    convertible to Event
}

/// The `FocusEvent` is fired when an element has received focus. The main
/// difference between this event and focusin is that only the latter bubbles.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/focus)
pub struct FocusEvent( Reference );

impl IEvent for FocusEvent {}
impl IFocusEvent for FocusEvent {}
impl ConcreteEvent for FocusEvent {
    const EVENT_TYPE: &'static str = "focus";
}

reference_boilerplate! {
    FocusEvent,
    instanceof FocusEvent
    convertible to Event
    convertible to FocusRelatedEvent
}

/// The `BlurEvent` is fired when an element has lost focus. The main difference
/// between this event and focusout is that only the latter bubbles.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/blur)
pub struct BlurEvent( Reference );

impl IEvent for BlurEvent {}
impl IFocusEvent for BlurEvent {}
impl ConcreteEvent for BlurEvent {
    const EVENT_TYPE: &'static str = "blur";
}

reference_boilerplate! {
    BlurEvent,
    instanceof FocusEvent
    convertible to Event
    convertible to FocusRelatedEvent
}

/// The `HashChangeEvent` is fired when the fragment
/// identifier of the URL has changed (the part of the URL
/// that follows the # symbol, including the # symbol).
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/hashchange)
pub struct HashChangeEvent( Reference );

impl IEvent for HashChangeEvent {}
impl ConcreteEvent for HashChangeEvent {
    const EVENT_TYPE: &'static str = "hashchange";
}

reference_boilerplate! {
    HashChangeEvent,
    instanceof HashChangeEvent
    convertible to Event
}

impl HashChangeEvent {
    /// The previous URL from which the window was navigated.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/HashChangeEvent)
    #[inline]
    pub fn old_url( &self ) -> String {
        js!(
            return @{self.as_ref()}.oldURL;
        ).try_into().unwrap()
    }

    /// The new URL to which the window was navigated.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/HashChangeEvent)
    #[inline]
    pub fn new_url( &self ) -> String {
        js!(
            return @{self.as_ref()}.newURL;
        ).try_into().unwrap()
    }
}

/// The `IProgressEvent` interface represents progress-related
/// events.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/ProgressEvent)
pub trait IProgressEvent: IEvent {
    /// Indicates whether the progress is measureable.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/ProgressEvent/lengthComputable)
    #[inline]
    fn length_computable( &self ) -> bool {
        js!(
            return @{self.as_ref()}.lengthComputable;
        ).try_into().unwrap()
    }

    /// Returns the amount of work already performed by the underlying process.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/ProgressEvent/loaded)
    #[inline]
    fn loaded( &self ) -> u64 {
        js!(
            return @{self.as_ref()}.loaded;
        ).try_into().unwrap()
    }

    /// Returns the total amount of work that the underlying process will perform.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/ProgressEvent/total)
    #[inline]
    fn total( &self ) -> u64 {
        js!(
            return @{self.as_ref()}.total;
        ).try_into().unwrap()
    }
}

/// A reference to a JavaScript object which implements the [IProgressEvent](trait.IProgressEvent.html)
/// interface.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/ProgressEvent)
pub struct ProgressRelatedEvent( Reference );

impl IEvent for ProgressRelatedEvent {}
impl IProgressEvent for ProgressRelatedEvent {}

reference_boilerplate! {
    ProgressRelatedEvent,
    instanceof ProgressEvent
    convertible to Event
}

/// The `ProgressEvent` is fired to indicate that an operation is in progress.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/progress)
pub struct ProgressEvent( Reference );

impl IEvent for ProgressEvent {}
impl IProgressEvent for ProgressEvent {}
impl ConcreteEvent for ProgressEvent {
    const EVENT_TYPE: &'static str = "progress";
}

reference_boilerplate! {
    ProgressEvent,
    instanceof ProgressEvent
    convertible to Event
    convertible to ProgressRelatedEvent
}

/// The `LoadStartEvent` is fired when progress has begun on the loading of a resource.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/loadstart)
pub struct LoadStartEvent( Reference );

impl IEvent for LoadStartEvent {}
impl IProgressEvent for LoadStartEvent {}
impl ConcreteEvent for LoadStartEvent {
    const EVENT_TYPE: &'static str = "loadstart";
}

reference_boilerplate! {
    LoadStartEvent,
    instanceof ProgressEvent
    convertible to Event
    convertible to ProgressRelatedEvent
}

/// The `LoadEndEvent` is fired when progress has stopped on the loading of a resource,
/// e.g. after `ErrorEvent`, `AbortEvent` or `LoadEvent` have been dispatched.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/loadend)
pub struct LoadEndEvent( Reference );

impl IEvent for LoadEndEvent {}
impl IProgressEvent for LoadEndEvent {}
impl ConcreteEvent for LoadEndEvent {
    const EVENT_TYPE: &'static str = "loadend";
}

reference_boilerplate! {
    LoadEndEvent,
    instanceof ProgressEvent
    convertible to Event
    convertible to ProgressRelatedEvent
}

/// The `AbortEvent` is fired when the loading of a resource has been aborted.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/abort)
pub struct AbortEvent( Reference );

// TODO: This event is sometimes an UiEvent; what to do here?
impl IEvent for AbortEvent {}
impl ConcreteEvent for AbortEvent {
    const EVENT_TYPE: &'static str = "abort";
}

reference_boilerplate! {
    AbortEvent,
    instanceof Event
    convertible to Event
}

/// The `ErrorEvent` is fired when an error occurred; the exact circumstances vary,
/// since this event is used from a variety of APIs.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/error)
pub struct ErrorEvent( Reference );

// TODO: This event is sometimes an UiEvent; what to do here?
impl IEvent for ErrorEvent {}
impl ConcreteEvent for ErrorEvent {
    const EVENT_TYPE: &'static str = "error";
}

reference_boilerplate! {
    ErrorEvent,
    instanceof Event
    convertible to Event
}

impl ErrorEvent {
    /// Returns a human-readable error message describing the problem.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/ErrorEvent/message)
    #[inline]
    pub fn message( &self ) -> String {
        return js!(
            return @{self.as_ref()}.message;
        ).try_into().unwrap()
    }

    /// Returns the name of the script where the error occurred.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/ErrorEvent/filename)
    #[inline]
    pub fn filename( &self ) -> String {
        return js!(
            return @{self.as_ref()}.filename;
        ).try_into().unwrap()
    }

    /// Returns the line number of the script file where the error occurred.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/ErrorEvent/lineNo)
    #[inline]
    pub fn lineno( &self ) -> u32 {
        return js!(
            return @{self.as_ref()}.lineno;
        ).try_into().unwrap()
    }

    /// Returns the column number of the script file where the error occurred.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/ErrorEvent/colNo)
    #[inline]
    pub fn colno( &self ) -> u32 {
        return js!(
            return @{self.as_ref()}.colno;
        ).try_into().unwrap()
    }
}

#[cfg(web_api_tests)]
mod tests {
    use super::*;

    #[test]
    fn test_event() {
        let event: Event = js!(
            return new Event("dummy")
        ).try_into().unwrap();

        assert_eq!( event.event_type(), "dummy" );
        assert_eq!( event.bubbles(), false );
        assert!( !event.cancel_bubble() );
        assert!( !event.cancelable(), false );
        assert!( event.current_target().is_none() );
        assert!( !event.default_prevented() );
        assert_eq!( event.event_phase(), EventPhase::None );
        assert!( event.target().is_none() );
        assert!( event.time_stamp().is_some() );
        assert!( !event.is_trusted() );

        event.stop_immediate_propagation();
        event.stop_propagation();
    }

    #[test]
    fn test_change_event() {
        let event: ChangeEvent = js!(
            return new Event( @{ChangeEvent::EVENT_TYPE} );
        ).try_into().unwrap();
        assert_eq!( event.event_type(), ChangeEvent::EVENT_TYPE );
    }

    #[test]
    fn test_ui_event() {
        let event: UiEvent = js!(
            return new UIEvent(
                @{ClickEvent::EVENT_TYPE},
                {
                    detail: 1,
                }
            )
        ).try_into().unwrap();
        assert_eq!( event.event_type(), ClickEvent::EVENT_TYPE );
        assert_eq!( event.detail(), 1 );
        assert!( event.view().is_none() );
    }

    #[test]
    fn test_load_event() {
        let event: LoadEvent = js!(
            return new UIEvent( @{LoadEvent::EVENT_TYPE} );
        ).try_into().unwrap();
        assert_eq!( event.event_type(), LoadEvent::EVENT_TYPE );
    }

    #[test]
    fn test_mouse_event() {
        let event: MouseEvent = js!(
            return new MouseEvent(
                @{ClickEvent::EVENT_TYPE},
                {
                    altKey: false,
                    button: 2,
                    buttons: 6,
                    clientX: 3.0,
                    clientY: 4.0,
                    ctrlKey: true,
                    metaKey: false,
                    screenX: 1.0,
                    screenY: 2.0,
                    shiftKey: true
                }
            );
        ).try_into().unwrap();
        assert_eq!( event.event_type(), ClickEvent::EVENT_TYPE );
        assert_eq!( event.alt_key(), false );
        assert_eq!( event.button(), MouseButton::Right );
        assert!( !event.buttons().is_down( MouseButton::Left ) );
        assert!( event.buttons().is_down( MouseButton::Right ) );
        assert!( event.buttons().is_down( MouseButton::Wheel ) );
        assert_eq!( event.client_x(), 3.0 );
        assert_eq!( event.client_y(), 4.0 );
        assert!( event.ctrl_key() );
        assert!( !event.get_modifier_state( ModifierKey::Alt ) );
        assert!( event.get_modifier_state( ModifierKey::Ctrl ) );
        assert!( event.get_modifier_state( ModifierKey::Shift ) );
        assert!( !event.meta_key() );
        assert_eq!( event.movement_x(), 0.0 );
        assert_eq!( event.movement_y(), 0.0 );
        assert!( event.region().is_none() );
        assert!( event.related_target().is_none() );
        assert_eq!( event.screen_x(), 1.0 );
        assert_eq!( event.screen_y(), 2.0 );
        assert!( event.shift_key() );
    }

    #[test]
    fn test_click_event() {
        let event: ClickEvent = js!(
            return new MouseEvent( @{ClickEvent::EVENT_TYPE} );
        ).try_into().unwrap();
        assert_eq!( event.event_type(), ClickEvent::EVENT_TYPE );
    }

    #[test]
    fn test_double_click_event() {
        let event: DoubleClickEvent = js!(
            return new MouseEvent( @{DoubleClickEvent::EVENT_TYPE} );
        ).try_into().unwrap();
        assert_eq!( event.event_type(), DoubleClickEvent::EVENT_TYPE );
    }

    #[test]
    fn test_keyboard_event() {
        let event: KeyboardEvent = js!(
            return new KeyboardEvent(
                @{KeypressEvent::EVENT_TYPE},
                {
                    key: "A",
                    code: "KeyA",
                    location: 0,
                    ctrlKey: true,
                    shiftKey: false,
                    altKey: true,
                    metaKey: false,
                    repeat: true,
                    isComposing: false
                }
            );
        ).try_into().unwrap();
        assert!( event.alt_key() );
        assert_eq!( event.code(), "KeyA" );
        assert!( event.ctrl_key() );
        assert!( event.get_modifier_state( ModifierKey::Alt ) );
        assert!( event.get_modifier_state( ModifierKey::Ctrl ) );
        assert!( !event.get_modifier_state( ModifierKey::Shift ) );
        assert!( !event.is_composing() );
        assert_eq!( event.location(), KeyboardLocation::Standard );
        assert_eq!( event.key(), "A" );
        assert!( !event.meta_key() );
        assert!( event.repeat() );
        assert!( !event.shift_key() );
    }

    #[test]
    fn test_keypress_event() {
        let event: KeypressEvent = js!(
            return new KeyboardEvent( @{KeypressEvent::EVENT_TYPE} );
        ).try_into().unwrap();
        assert_eq!( event.event_type(), KeypressEvent::EVENT_TYPE );
    }

    #[test]
    fn test_focus_event() {
        let event: FocusEvent = js!(
            return new FocusEvent( "focus" );
        ).try_into().unwrap();
        assert_eq!( event.event_type(), "focus" );
        assert!( event.related_target().is_none() );
    }

    #[test]
    fn test_blur_event() {
        let event: BlurEvent = js!(
            return new FocusEvent( @{BlurEvent::EVENT_TYPE} );
        ).try_into().unwrap();
        assert_eq!( event.event_type(), BlurEvent::EVENT_TYPE );
    }

    #[test]
    fn test_hash_change_event() {
        let event: HashChangeEvent = js!(
            return new HashChangeEvent(
                @{HashChangeEvent::EVENT_TYPE},
                {
                    oldURL: "http://test.com#foo",
                    newURL: "http://test.com#bar"
                }
            );
        ).try_into().unwrap();
        assert_eq!( event.event_type(), HashChangeEvent::EVENT_TYPE );
        assert_eq!( event.old_url(), "http://test.com#foo" );
        assert_eq!( event.new_url(), "http://test.com#bar" );
    }

    #[test]
    fn test_progress_event() {
        let event: ProgressEvent = js!(
            return new ProgressEvent(
                @{ProgressEvent::EVENT_TYPE},
                {
                    lengthComputable: true,
                    loaded: 10,
                    total: 100,
                }
            );
        ).try_into().unwrap();
        assert_eq!( event.event_type(), ProgressEvent::EVENT_TYPE );
        assert!( event.length_computable() );
        assert_eq!( event.loaded(), 10 );
        assert_eq!( event.total(), 100 );
    }

    #[test]
    fn test_load_start_event() {
        let event: LoadStartEvent = js!(
            return new ProgressEvent( @{LoadStartEvent::EVENT_TYPE} );
        ).try_into().unwrap();
        assert_eq!( event.event_type(), LoadStartEvent::EVENT_TYPE );
    }

    #[test]
    fn test_load_end_event() {
        let event: LoadEndEvent = js!(
            return new ProgressEvent( @{LoadEndEvent::EVENT_TYPE} );
        ).try_into().unwrap();
        assert_eq!( event.event_type(), LoadEndEvent::EVENT_TYPE );
    }

    #[test]
    fn test_abort_event() {
        let event: AbortEvent = js!(
            return new Event( @{AbortEvent::EVENT_TYPE} );
        ).try_into().unwrap();
        assert_eq!( event.event_type(), AbortEvent::EVENT_TYPE );
    }

    #[test]
    fn test_error_event() {
        let event: ErrorEvent = js!(
            return new ErrorEvent(
                @{ErrorEvent::EVENT_TYPE},
                {
                    message: "Dummy error",
                    filename: "Dummy.js",
                    lineno: 5,
                    colno: 10
                }
            );
        ).try_into().unwrap();
        assert_eq!( event.event_type(), ErrorEvent::EVENT_TYPE );
        assert_eq!( event.message(), "Dummy error".to_string() );
        assert_eq!( event.filename(), "Dummy.js".to_string() );
        assert_eq!( event.lineno(), 5 );
        assert_eq!( event.colno(), 10 );
    }
}
