use webcore::value::{Reference, Value};
use webcore::try_from::{TryFrom, TryInto};

/// The `IEvent` interface represents any event which takes place in the DOM; some
/// are user-generated (such as mouse or keyboard events), while others are
/// generated by APIs (such as events that indicate an animation has finished
/// running, a video has been paused, and so forth). There are many types of event,
/// some of which use other interfaces based on the main `IEvent` interface. `IEvent`
/// itself contains the properties and methods which are common to all events.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event)
pub trait IEvent: AsRef< Reference > + TryFrom< Value > {
    /// Returns a string containing the type of event. It is set when
    /// the event is constructed and is the name commonly used to refer
    /// to the specific event.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event/type)
    fn event_type( &self ) -> String {
        js!( return @{self.as_ref()}.type; ).try_into().unwrap()
    }

    /// Cancels the event if it is cancelable, without
    /// stopping further propagation of the event.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event/preventDefault)
    fn prevent_default( &self ) {
        js! { @(no_return)
            @{self.as_ref()}.preventDefault();
        }
    }
}

pub trait ConcreteEvent: IEvent {
    // TODO: Switch to an associated constant for `event_type` once they stabilize.

    /// Returns a string representing the event type.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event/type)
    fn static_event_type() -> &'static str;
}

/// A reference to a JavaScript object which implements the [IEvent](trait.IEvent.html)
/// interface.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/Event)
pub struct Event( Reference );

impl IEvent for Event {}

reference_boilerplate! {
    Event,
    instanceof Event
}

/// The `ChangeEvent` is fired for input, select, and textarea
/// elements when a change to the element's value is committed
/// by the user. Unlike the input event, the change event is not
/// necessarily fired for each change to an element's value.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/change)
pub struct ChangeEvent( Reference );

impl IEvent for ChangeEvent {}
impl ConcreteEvent for ChangeEvent {
    #[inline]
    fn static_event_type() -> &'static str {
        "change"
    }
}

reference_boilerplate! {
    ChangeEvent,
    instanceof Event
    convertible to Event
}

/// The `IUiEvent` interface represents simple user interface events.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/UIEvent)
pub trait IUiEvent: IEvent {
}

/// A reference to a JavaScript object which implements the [IUiEvent](trait.IUiEvent.html)
/// interface.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/UIEvent)
pub struct UiEvent( Reference );

impl IEvent for UiEvent {}
impl IUiEvent for UiEvent {}

reference_boilerplate! {
    UiEvent,
    instanceof UiEvent
    convertible to Event
}

/// The `IMouseEvent` interface represents events that occur due to the user
/// interacting with a pointing device (such as a mouse).
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent)
pub trait IMouseEvent: IUiEvent {
}

/// A reference to a JavaScript object which implements the [IMouseEvent](trait.IMouseEvent.html)
/// interface.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/MouseEvent)
pub struct MouseEvent( Reference );

impl IEvent for MouseEvent {}
impl IUiEvent for MouseEvent {}
impl IMouseEvent for MouseEvent {}

reference_boilerplate! {
    MouseEvent,
    instanceof MouseEvent
    convertible to Event
    convertible to UiEvent
}

/// The `ClickEvent` is fired when a pointing device button (usually a
/// mouse's primary button) is pressed and released on a single element.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/click)
pub struct ClickEvent( Reference );

impl IEvent for ClickEvent {}
impl IUiEvent for ClickEvent {}
impl IMouseEvent for ClickEvent {}
impl ConcreteEvent for ClickEvent {
    #[inline]
    fn static_event_type() -> &'static str {
        "click"
    }
}

reference_boilerplate! {
    ClickEvent,
    instanceof MouseEvent
    convertible to Event
    convertible to UiEvent
    convertible to MouseEvent
}

/// The `DoubleClickEvent` is fired when a pointing device button
/// (usually a mouse's primary button) is clicked twice on a single
/// element.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/dblclick)
pub struct DoubleClickEvent( Reference );

impl IEvent for DoubleClickEvent {}
impl IUiEvent for DoubleClickEvent {}
impl IMouseEvent for DoubleClickEvent {}
impl ConcreteEvent for DoubleClickEvent {
    #[inline]
    fn static_event_type() -> &'static str {
        "dblclick"
    }
}

reference_boilerplate! {
    DoubleClickEvent,
    instanceof MouseEvent
    convertible to Event
    convertible to UiEvent
    convertible to MouseEvent
}

/// `IKeyboardEvent` objects describe a user interaction with the
/// keyboard. Each event describes a key; the event type identifies
/// what kind of activity was performed.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent)
pub trait IKeyboardEvent: IEvent {
    /// Returns the value of a key or keys pressed by the user.
    ///
    /// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent/key)
    fn key( &self ) -> String {
        js!( return @{self.as_ref()}.key; ).into_string().unwrap()
    }
}

/// A reference to a JavaScript object which implements the [IKeyboardEvent](trait.IKeyboardEvent.html)
/// interface.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/KeyboardEvent)
pub struct KeyboardEvent( Reference );

impl IEvent for KeyboardEvent {}
impl IKeyboardEvent for KeyboardEvent {}

reference_boilerplate! {
    KeyboardEvent,
    instanceof KeyboardEvent
    convertible to Event
}

/// The `KeypressEvent` is fired when a key is pressed down, and that
/// key normally produces a character value.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/keypress)
pub struct KeypressEvent( Reference );

impl IEvent for KeypressEvent {}
impl IKeyboardEvent for KeypressEvent {}
impl ConcreteEvent for KeypressEvent {
    #[inline]
    fn static_event_type() -> &'static str {
        "keypress"
    }
}

reference_boilerplate! {
    KeypressEvent,
    instanceof KeyboardEvent
    convertible to Event
    convertible to KeyboardEvent
}

/// The `IFocusEvent` interface represents focus-related
/// events.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/FocusEvent)
pub trait IFocusEvent: IEvent {
}

/// A reference to a JavaScript object which implements the [IFocusEvent](trait.IFocusEvent.html)
/// interface.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/API/FocusEvent)
pub struct FocusRelatedEvent( Reference );

impl IEvent for FocusRelatedEvent {}
impl IFocusEvent for FocusRelatedEvent {}

reference_boilerplate! {
    FocusRelatedEvent,
    instanceof FocusEvent
    convertible to Event
}

/// The `FocusEvent` is fired when an element has received focus. The main
/// difference between this event and focusin is that only the latter bubbles.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/focus)
pub struct FocusEvent( Reference );

impl IEvent for FocusEvent {}
impl IFocusEvent for FocusEvent {}
impl ConcreteEvent for FocusEvent {
    #[inline]
    fn static_event_type() -> &'static str {
        "focus"
    }
}

reference_boilerplate! {
    FocusEvent,
    instanceof FocusEvent
    convertible to Event
    convertible to FocusRelatedEvent
}

/// The `BlurEvent` is fired when an element has lost focus. The main difference
/// between this event and focusout is that only the latter bubbles.
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/blur)
pub struct BlurEvent( Reference );

impl IEvent for BlurEvent {}
impl IFocusEvent for BlurEvent {}
impl ConcreteEvent for BlurEvent {
    #[inline]
    fn static_event_type() -> &'static str {
        "blur"
    }
}

reference_boilerplate! {
    BlurEvent,
    instanceof FocusEvent
    convertible to Event
    convertible to FocusRelatedEvent
}

/// The `HashChangeEvent` is fired when the fragment
/// identifier of the URL has changed (the part of the URL
/// that follows the # symbol, including the # symbol).
///
/// [(JavaScript docs)](https://developer.mozilla.org/en-US/docs/Web/Events/hashchange)
pub struct HashChangeEvent( Reference );

impl IEvent for HashChangeEvent {}
impl ConcreteEvent for HashChangeEvent {
    #[inline]
    fn static_event_type() -> &'static str {
        "hashchange"
    }
}

reference_boilerplate! {
    HashChangeEvent,
    instanceof HashChangeEvent
    convertible to Event
}
