//-------------------------------------------------------------------------
// @file matrix6x6.rs
//
// @date 06/02/20 20:59:42
// @author Martin Noblia
// @email mnoblia@disroot.org
//
// @brief
//
// @detail
//
// Licence MIT:
// Copyright <2020> <Martin Noblia>
//
// Permission is hereby granted, free of charge, to any person obtaining a copy
// of this software and associated documentation files (the "Software"), to deal
// in the Software without restriction, including without limitation the rights
// to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
// copies of the Software, and to permit persons to whom the Software is
// furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.  THE SOFTWARE IS PROVIDED
// "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR IMPLIED, INCLUDING BUT NOT
// LIMITED TO THE WARRANTIES OF MERCHANTABILITY, FITNESS FOR A PARTICULAR
// PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
// HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN
// ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION
// WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
//-------------------------------------------------------------------------
use std::fmt;
use std::ops::{Add, Mul, Div, Sub, AddAssign, SubAssign, Neg};
use std::ops::{Deref, DerefMut, Index, IndexMut};

use num::{Float, Num, One, Zero, Signed};

use crate::slices_methods::*;
use crate::matrix5x5::M55;
use crate::vector6::V6;
use crate::matrix3x3::M33;
use crate::traits::LinearAlgebra;
use crate::utils::nearly_zero;
//-------------------------------------------------------------------------
//                        code
//-------------------------------------------------------------------------
/// A static Matrix of 6x6 shape
#[derive(Copy, Clone, Debug, PartialEq)]
pub struct M66<T>([[T; 6]; 6]);

impl<T> M66<T> {
    pub fn new(data_input: [[T; 6]; 6]) -> Self {
        Self(data_input)
    }
    pub fn rows(&self) -> usize {
        self.0.len()
    }
    pub fn cols(&self) -> usize {
        self.rows()
    }
}


impl<T: Float + std::iter::Sum> LinearAlgebra<T> for M66<T> {
    #[inline]
    fn rows(&self) -> usize {
        self.0.len()
    }

    #[inline]
    fn cols(&self) -> usize {
        self.rows()
    }

    #[inline]
    fn transpose(&self) -> Self {
        let a_00 = self[(0, 0)];
        let a_01 = self[(0, 1)];
        let a_02 = self[(0, 2)];
        let a_03 = self[(0, 3)];
        let a_04 = self[(0, 4)];
        let a_05 = self[(0, 5)];
        let a_10 = self[(1, 0)];
        let a_11 = self[(1, 1)];
        let a_12 = self[(1, 2)];
        let a_13 = self[(1, 3)];
        let a_14 = self[(1, 4)];
        let a_15 = self[(1, 5)];
        let a_20 = self[(2, 0)];
        let a_21 = self[(2, 1)];
        let a_22 = self[(2, 2)];
        let a_23 = self[(2, 3)];
        let a_24 = self[(2, 4)];
        let a_25 = self[(2, 5)];
        let a_30 = self[(3, 0)];
        let a_31 = self[(3, 1)];
        let a_32 = self[(3, 2)];
        let a_33 = self[(3, 3)];
        let a_34 = self[(3, 4)];
        let a_35 = self[(3, 5)];
        let a_40 = self[(4, 0)];
        let a_41 = self[(4, 1)];
        let a_42 = self[(4, 2)];
        let a_43 = self[(4, 3)];
        let a_44 = self[(4, 4)];
        let a_45 = self[(4, 5)];
        let a_50 = self[(5, 0)];
        let a_51 = self[(5, 1)];
        let a_52 = self[(5, 2)];
        let a_53 = self[(5, 3)];
        let a_54 = self[(5, 4)];
        let a_55 = self[(5, 5)];

        M66::new([
            [a_00, a_10, a_20, a_30, a_40, a_50],
            [a_01, a_11, a_21, a_31, a_41, a_51],
            [a_02, a_12, a_22, a_32, a_42, a_52],
            [a_03, a_13, a_23, a_33, a_43, a_53],
            [a_04, a_14, a_24, a_34, a_44, a_54],
            [a_05, a_15, a_25, a_35, a_45, a_55],
        ])
    }

    #[inline(always)]
    fn trace(&self) -> T {
           self[(0, 0)] + self[(1, 1)] + self[(2, 2)] + self[(3, 3)] + self[(4, 4)] + self[(5, 5)]
    }

    fn norm2(&self) -> T {
        T::sqrt(
            self.iter()
                .flatten()
                .cloned()
                .map(|element| element * element)
                .sum(),
        )
    }

    #[inline(always)]
    fn det(&self) -> T {
        let a_00 = self[(0, 0)];
        let a_01 = self[(0, 1)];
        let a_02 = self[(0, 2)];
        let a_03 = self[(0, 3)];
        let a_04 = self[(0, 4)];
        let a_05 = self[(0, 5)];
        let a_10 = self[(1, 0)];
        let a_11 = self[(1, 1)];
        let a_12 = self[(1, 2)];
        let a_13 = self[(1, 3)];
        let a_14 = self[(1, 4)];
        let a_15 = self[(1, 5)];
        let a_20 = self[(2, 0)];
        let a_21 = self[(2, 1)];
        let a_22 = self[(2, 2)];
        let a_23 = self[(2, 3)];
        let a_24 = self[(2, 4)];
        let a_25 = self[(2, 5)];
        let a_30 = self[(3, 0)];
        let a_31 = self[(3, 1)];
        let a_32 = self[(3, 2)];
        let a_33 = self[(3, 3)];
        let a_34 = self[(3, 4)];
        let a_35 = self[(3, 5)];
        let a_40 = self[(4, 0)];
        let a_41 = self[(4, 1)];
        let a_42 = self[(4, 2)];
        let a_43 = self[(4, 3)];
        let a_44 = self[(4, 4)];
        let a_45 = self[(4, 5)];
        let a_50 = self[(5, 0)];
        let a_51 = self[(5, 1)];
        let a_52 = self[(5, 2)];
        let a_53 = self[(5, 3)];
        let a_54 = self[(5, 4)];
        let a_55 = self[(5, 5)];

        a_00 * a_11 * a_22 * a_33 * a_44 * a_55
            - a_00 * a_11 * a_22 * a_33 * a_45 * a_54
            - a_00 * a_11 * a_22 * a_34 * a_43 * a_55
            + a_00 * a_11 * a_22 * a_34 * a_45 * a_53
            + a_00 * a_11 * a_22 * a_35 * a_43 * a_54
            - a_00 * a_11 * a_22 * a_35 * a_44 * a_53
            - a_00 * a_11 * a_23 * a_32 * a_44 * a_55
            + a_00 * a_11 * a_23 * a_32 * a_45 * a_54
            + a_00 * a_11 * a_23 * a_34 * a_42 * a_55
            - a_00 * a_11 * a_23 * a_34 * a_45 * a_52
            - a_00 * a_11 * a_23 * a_35 * a_42 * a_54
            + a_00 * a_11 * a_23 * a_35 * a_44 * a_52
            + a_00 * a_11 * a_24 * a_32 * a_43 * a_55
            - a_00 * a_11 * a_24 * a_32 * a_45 * a_53
            - a_00 * a_11 * a_24 * a_33 * a_42 * a_55
            + a_00 * a_11 * a_24 * a_33 * a_45 * a_52
            + a_00 * a_11 * a_24 * a_35 * a_42 * a_53
            - a_00 * a_11 * a_24 * a_35 * a_43 * a_52
            - a_00 * a_11 * a_25 * a_32 * a_43 * a_54
            + a_00 * a_11 * a_25 * a_32 * a_44 * a_53
            + a_00 * a_11 * a_25 * a_33 * a_42 * a_54
            - a_00 * a_11 * a_25 * a_33 * a_44 * a_52
            - a_00 * a_11 * a_25 * a_34 * a_42 * a_53
            + a_00 * a_11 * a_25 * a_34 * a_43 * a_52
            - a_00 * a_12 * a_21 * a_33 * a_44 * a_55
            + a_00 * a_12 * a_21 * a_33 * a_45 * a_54
            + a_00 * a_12 * a_21 * a_34 * a_43 * a_55
            - a_00 * a_12 * a_21 * a_34 * a_45 * a_53
            - a_00 * a_12 * a_21 * a_35 * a_43 * a_54
            + a_00 * a_12 * a_21 * a_35 * a_44 * a_53
            + a_00 * a_12 * a_23 * a_31 * a_44 * a_55
            - a_00 * a_12 * a_23 * a_31 * a_45 * a_54
            - a_00 * a_12 * a_23 * a_34 * a_41 * a_55
            + a_00 * a_12 * a_23 * a_34 * a_45 * a_51
            + a_00 * a_12 * a_23 * a_35 * a_41 * a_54
            - a_00 * a_12 * a_23 * a_35 * a_44 * a_51
            - a_00 * a_12 * a_24 * a_31 * a_43 * a_55
            + a_00 * a_12 * a_24 * a_31 * a_45 * a_53
            + a_00 * a_12 * a_24 * a_33 * a_41 * a_55
            - a_00 * a_12 * a_24 * a_33 * a_45 * a_51
            - a_00 * a_12 * a_24 * a_35 * a_41 * a_53
            + a_00 * a_12 * a_24 * a_35 * a_43 * a_51
            + a_00 * a_12 * a_25 * a_31 * a_43 * a_54
            - a_00 * a_12 * a_25 * a_31 * a_44 * a_53
            - a_00 * a_12 * a_25 * a_33 * a_41 * a_54
            + a_00 * a_12 * a_25 * a_33 * a_44 * a_51
            + a_00 * a_12 * a_25 * a_34 * a_41 * a_53
            - a_00 * a_12 * a_25 * a_34 * a_43 * a_51
            + a_00 * a_13 * a_21 * a_32 * a_44 * a_55
            - a_00 * a_13 * a_21 * a_32 * a_45 * a_54
            - a_00 * a_13 * a_21 * a_34 * a_42 * a_55
            + a_00 * a_13 * a_21 * a_34 * a_45 * a_52
            + a_00 * a_13 * a_21 * a_35 * a_42 * a_54
            - a_00 * a_13 * a_21 * a_35 * a_44 * a_52
            - a_00 * a_13 * a_22 * a_31 * a_44 * a_55
            + a_00 * a_13 * a_22 * a_31 * a_45 * a_54
            + a_00 * a_13 * a_22 * a_34 * a_41 * a_55
            - a_00 * a_13 * a_22 * a_34 * a_45 * a_51
            - a_00 * a_13 * a_22 * a_35 * a_41 * a_54
            + a_00 * a_13 * a_22 * a_35 * a_44 * a_51
            + a_00 * a_13 * a_24 * a_31 * a_42 * a_55
            - a_00 * a_13 * a_24 * a_31 * a_45 * a_52
            - a_00 * a_13 * a_24 * a_32 * a_41 * a_55
            + a_00 * a_13 * a_24 * a_32 * a_45 * a_51
            + a_00 * a_13 * a_24 * a_35 * a_41 * a_52
            - a_00 * a_13 * a_24 * a_35 * a_42 * a_51
            - a_00 * a_13 * a_25 * a_31 * a_42 * a_54
            + a_00 * a_13 * a_25 * a_31 * a_44 * a_52
            + a_00 * a_13 * a_25 * a_32 * a_41 * a_54
            - a_00 * a_13 * a_25 * a_32 * a_44 * a_51
            - a_00 * a_13 * a_25 * a_34 * a_41 * a_52
            + a_00 * a_13 * a_25 * a_34 * a_42 * a_51
            - a_00 * a_14 * a_21 * a_32 * a_43 * a_55
            + a_00 * a_14 * a_21 * a_32 * a_45 * a_53
            + a_00 * a_14 * a_21 * a_33 * a_42 * a_55
            - a_00 * a_14 * a_21 * a_33 * a_45 * a_52
            - a_00 * a_14 * a_21 * a_35 * a_42 * a_53
            + a_00 * a_14 * a_21 * a_35 * a_43 * a_52
            + a_00 * a_14 * a_22 * a_31 * a_43 * a_55
            - a_00 * a_14 * a_22 * a_31 * a_45 * a_53
            - a_00 * a_14 * a_22 * a_33 * a_41 * a_55
            + a_00 * a_14 * a_22 * a_33 * a_45 * a_51
            + a_00 * a_14 * a_22 * a_35 * a_41 * a_53
            - a_00 * a_14 * a_22 * a_35 * a_43 * a_51
            - a_00 * a_14 * a_23 * a_31 * a_42 * a_55
            + a_00 * a_14 * a_23 * a_31 * a_45 * a_52
            + a_00 * a_14 * a_23 * a_32 * a_41 * a_55
            - a_00 * a_14 * a_23 * a_32 * a_45 * a_51
            - a_00 * a_14 * a_23 * a_35 * a_41 * a_52
            + a_00 * a_14 * a_23 * a_35 * a_42 * a_51
            + a_00 * a_14 * a_25 * a_31 * a_42 * a_53
            - a_00 * a_14 * a_25 * a_31 * a_43 * a_52
            - a_00 * a_14 * a_25 * a_32 * a_41 * a_53
            + a_00 * a_14 * a_25 * a_32 * a_43 * a_51
            + a_00 * a_14 * a_25 * a_33 * a_41 * a_52
            - a_00 * a_14 * a_25 * a_33 * a_42 * a_51
            + a_00 * a_15 * a_21 * a_32 * a_43 * a_54
            - a_00 * a_15 * a_21 * a_32 * a_44 * a_53
            - a_00 * a_15 * a_21 * a_33 * a_42 * a_54
            + a_00 * a_15 * a_21 * a_33 * a_44 * a_52
            + a_00 * a_15 * a_21 * a_34 * a_42 * a_53
            - a_00 * a_15 * a_21 * a_34 * a_43 * a_52
            - a_00 * a_15 * a_22 * a_31 * a_43 * a_54
            + a_00 * a_15 * a_22 * a_31 * a_44 * a_53
            + a_00 * a_15 * a_22 * a_33 * a_41 * a_54
            - a_00 * a_15 * a_22 * a_33 * a_44 * a_51
            - a_00 * a_15 * a_22 * a_34 * a_41 * a_53
            + a_00 * a_15 * a_22 * a_34 * a_43 * a_51
            + a_00 * a_15 * a_23 * a_31 * a_42 * a_54
            - a_00 * a_15 * a_23 * a_31 * a_44 * a_52
            - a_00 * a_15 * a_23 * a_32 * a_41 * a_54
            + a_00 * a_15 * a_23 * a_32 * a_44 * a_51
            + a_00 * a_15 * a_23 * a_34 * a_41 * a_52
            - a_00 * a_15 * a_23 * a_34 * a_42 * a_51
            - a_00 * a_15 * a_24 * a_31 * a_42 * a_53
            + a_00 * a_15 * a_24 * a_31 * a_43 * a_52
            + a_00 * a_15 * a_24 * a_32 * a_41 * a_53
            - a_00 * a_15 * a_24 * a_32 * a_43 * a_51
            - a_00 * a_15 * a_24 * a_33 * a_41 * a_52
            + a_00 * a_15 * a_24 * a_33 * a_42 * a_51
            - a_01 * a_10 * a_22 * a_33 * a_44 * a_55
            + a_01 * a_10 * a_22 * a_33 * a_45 * a_54
            + a_01 * a_10 * a_22 * a_34 * a_43 * a_55
            - a_01 * a_10 * a_22 * a_34 * a_45 * a_53
            - a_01 * a_10 * a_22 * a_35 * a_43 * a_54
            + a_01 * a_10 * a_22 * a_35 * a_44 * a_53
            + a_01 * a_10 * a_23 * a_32 * a_44 * a_55
            - a_01 * a_10 * a_23 * a_32 * a_45 * a_54
            - a_01 * a_10 * a_23 * a_34 * a_42 * a_55
            + a_01 * a_10 * a_23 * a_34 * a_45 * a_52
            + a_01 * a_10 * a_23 * a_35 * a_42 * a_54
            - a_01 * a_10 * a_23 * a_35 * a_44 * a_52
            - a_01 * a_10 * a_24 * a_32 * a_43 * a_55
            + a_01 * a_10 * a_24 * a_32 * a_45 * a_53
            + a_01 * a_10 * a_24 * a_33 * a_42 * a_55
            - a_01 * a_10 * a_24 * a_33 * a_45 * a_52
            - a_01 * a_10 * a_24 * a_35 * a_42 * a_53
            + a_01 * a_10 * a_24 * a_35 * a_43 * a_52
            + a_01 * a_10 * a_25 * a_32 * a_43 * a_54
            - a_01 * a_10 * a_25 * a_32 * a_44 * a_53
            - a_01 * a_10 * a_25 * a_33 * a_42 * a_54
            + a_01 * a_10 * a_25 * a_33 * a_44 * a_52
            + a_01 * a_10 * a_25 * a_34 * a_42 * a_53
            - a_01 * a_10 * a_25 * a_34 * a_43 * a_52
            + a_01 * a_12 * a_20 * a_33 * a_44 * a_55
            - a_01 * a_12 * a_20 * a_33 * a_45 * a_54
            - a_01 * a_12 * a_20 * a_34 * a_43 * a_55
            + a_01 * a_12 * a_20 * a_34 * a_45 * a_53
            + a_01 * a_12 * a_20 * a_35 * a_43 * a_54
            - a_01 * a_12 * a_20 * a_35 * a_44 * a_53
            - a_01 * a_12 * a_23 * a_30 * a_44 * a_55
            + a_01 * a_12 * a_23 * a_30 * a_45 * a_54
            + a_01 * a_12 * a_23 * a_34 * a_40 * a_55
            - a_01 * a_12 * a_23 * a_34 * a_45 * a_50
            - a_01 * a_12 * a_23 * a_35 * a_40 * a_54
            + a_01 * a_12 * a_23 * a_35 * a_44 * a_50
            + a_01 * a_12 * a_24 * a_30 * a_43 * a_55
            - a_01 * a_12 * a_24 * a_30 * a_45 * a_53
            - a_01 * a_12 * a_24 * a_33 * a_40 * a_55
            + a_01 * a_12 * a_24 * a_33 * a_45 * a_50
            + a_01 * a_12 * a_24 * a_35 * a_40 * a_53
            - a_01 * a_12 * a_24 * a_35 * a_43 * a_50
            - a_01 * a_12 * a_25 * a_30 * a_43 * a_54
            + a_01 * a_12 * a_25 * a_30 * a_44 * a_53
            + a_01 * a_12 * a_25 * a_33 * a_40 * a_54
            - a_01 * a_12 * a_25 * a_33 * a_44 * a_50
            - a_01 * a_12 * a_25 * a_34 * a_40 * a_53
            + a_01 * a_12 * a_25 * a_34 * a_43 * a_50
            - a_01 * a_13 * a_20 * a_32 * a_44 * a_55
            + a_01 * a_13 * a_20 * a_32 * a_45 * a_54
            + a_01 * a_13 * a_20 * a_34 * a_42 * a_55
            - a_01 * a_13 * a_20 * a_34 * a_45 * a_52
            - a_01 * a_13 * a_20 * a_35 * a_42 * a_54
            + a_01 * a_13 * a_20 * a_35 * a_44 * a_52
            + a_01 * a_13 * a_22 * a_30 * a_44 * a_55
            - a_01 * a_13 * a_22 * a_30 * a_45 * a_54
            - a_01 * a_13 * a_22 * a_34 * a_40 * a_55
            + a_01 * a_13 * a_22 * a_34 * a_45 * a_50
            + a_01 * a_13 * a_22 * a_35 * a_40 * a_54
            - a_01 * a_13 * a_22 * a_35 * a_44 * a_50
            - a_01 * a_13 * a_24 * a_30 * a_42 * a_55
            + a_01 * a_13 * a_24 * a_30 * a_45 * a_52
            + a_01 * a_13 * a_24 * a_32 * a_40 * a_55
            - a_01 * a_13 * a_24 * a_32 * a_45 * a_50
            - a_01 * a_13 * a_24 * a_35 * a_40 * a_52
            + a_01 * a_13 * a_24 * a_35 * a_42 * a_50
            + a_01 * a_13 * a_25 * a_30 * a_42 * a_54
            - a_01 * a_13 * a_25 * a_30 * a_44 * a_52
            - a_01 * a_13 * a_25 * a_32 * a_40 * a_54
            + a_01 * a_13 * a_25 * a_32 * a_44 * a_50
            + a_01 * a_13 * a_25 * a_34 * a_40 * a_52
            - a_01 * a_13 * a_25 * a_34 * a_42 * a_50
            + a_01 * a_14 * a_20 * a_32 * a_43 * a_55
            - a_01 * a_14 * a_20 * a_32 * a_45 * a_53
            - a_01 * a_14 * a_20 * a_33 * a_42 * a_55
            + a_01 * a_14 * a_20 * a_33 * a_45 * a_52
            + a_01 * a_14 * a_20 * a_35 * a_42 * a_53
            - a_01 * a_14 * a_20 * a_35 * a_43 * a_52
            - a_01 * a_14 * a_22 * a_30 * a_43 * a_55
            + a_01 * a_14 * a_22 * a_30 * a_45 * a_53
            + a_01 * a_14 * a_22 * a_33 * a_40 * a_55
            - a_01 * a_14 * a_22 * a_33 * a_45 * a_50
            - a_01 * a_14 * a_22 * a_35 * a_40 * a_53
            + a_01 * a_14 * a_22 * a_35 * a_43 * a_50
            + a_01 * a_14 * a_23 * a_30 * a_42 * a_55
            - a_01 * a_14 * a_23 * a_30 * a_45 * a_52
            - a_01 * a_14 * a_23 * a_32 * a_40 * a_55
            + a_01 * a_14 * a_23 * a_32 * a_45 * a_50
            + a_01 * a_14 * a_23 * a_35 * a_40 * a_52
            - a_01 * a_14 * a_23 * a_35 * a_42 * a_50
            - a_01 * a_14 * a_25 * a_30 * a_42 * a_53
            + a_01 * a_14 * a_25 * a_30 * a_43 * a_52
            + a_01 * a_14 * a_25 * a_32 * a_40 * a_53
            - a_01 * a_14 * a_25 * a_32 * a_43 * a_50
            - a_01 * a_14 * a_25 * a_33 * a_40 * a_52
            + a_01 * a_14 * a_25 * a_33 * a_42 * a_50
            - a_01 * a_15 * a_20 * a_32 * a_43 * a_54
            + a_01 * a_15 * a_20 * a_32 * a_44 * a_53
            + a_01 * a_15 * a_20 * a_33 * a_42 * a_54
            - a_01 * a_15 * a_20 * a_33 * a_44 * a_52
            - a_01 * a_15 * a_20 * a_34 * a_42 * a_53
            + a_01 * a_15 * a_20 * a_34 * a_43 * a_52
            + a_01 * a_15 * a_22 * a_30 * a_43 * a_54
            - a_01 * a_15 * a_22 * a_30 * a_44 * a_53
            - a_01 * a_15 * a_22 * a_33 * a_40 * a_54
            + a_01 * a_15 * a_22 * a_33 * a_44 * a_50
            + a_01 * a_15 * a_22 * a_34 * a_40 * a_53
            - a_01 * a_15 * a_22 * a_34 * a_43 * a_50
            - a_01 * a_15 * a_23 * a_30 * a_42 * a_54
            + a_01 * a_15 * a_23 * a_30 * a_44 * a_52
            + a_01 * a_15 * a_23 * a_32 * a_40 * a_54
            - a_01 * a_15 * a_23 * a_32 * a_44 * a_50
            - a_01 * a_15 * a_23 * a_34 * a_40 * a_52
            + a_01 * a_15 * a_23 * a_34 * a_42 * a_50
            + a_01 * a_15 * a_24 * a_30 * a_42 * a_53
            - a_01 * a_15 * a_24 * a_30 * a_43 * a_52
            - a_01 * a_15 * a_24 * a_32 * a_40 * a_53
            + a_01 * a_15 * a_24 * a_32 * a_43 * a_50
            + a_01 * a_15 * a_24 * a_33 * a_40 * a_52
            - a_01 * a_15 * a_24 * a_33 * a_42 * a_50
            + a_02 * a_10 * a_21 * a_33 * a_44 * a_55
            - a_02 * a_10 * a_21 * a_33 * a_45 * a_54
            - a_02 * a_10 * a_21 * a_34 * a_43 * a_55
            + a_02 * a_10 * a_21 * a_34 * a_45 * a_53
            + a_02 * a_10 * a_21 * a_35 * a_43 * a_54
            - a_02 * a_10 * a_21 * a_35 * a_44 * a_53
            - a_02 * a_10 * a_23 * a_31 * a_44 * a_55
            + a_02 * a_10 * a_23 * a_31 * a_45 * a_54
            + a_02 * a_10 * a_23 * a_34 * a_41 * a_55
            - a_02 * a_10 * a_23 * a_34 * a_45 * a_51
            - a_02 * a_10 * a_23 * a_35 * a_41 * a_54
            + a_02 * a_10 * a_23 * a_35 * a_44 * a_51
            + a_02 * a_10 * a_24 * a_31 * a_43 * a_55
            - a_02 * a_10 * a_24 * a_31 * a_45 * a_53
            - a_02 * a_10 * a_24 * a_33 * a_41 * a_55
            + a_02 * a_10 * a_24 * a_33 * a_45 * a_51
            + a_02 * a_10 * a_24 * a_35 * a_41 * a_53
            - a_02 * a_10 * a_24 * a_35 * a_43 * a_51
            - a_02 * a_10 * a_25 * a_31 * a_43 * a_54
            + a_02 * a_10 * a_25 * a_31 * a_44 * a_53
            + a_02 * a_10 * a_25 * a_33 * a_41 * a_54
            - a_02 * a_10 * a_25 * a_33 * a_44 * a_51
            - a_02 * a_10 * a_25 * a_34 * a_41 * a_53
            + a_02 * a_10 * a_25 * a_34 * a_43 * a_51
            - a_02 * a_11 * a_20 * a_33 * a_44 * a_55
            + a_02 * a_11 * a_20 * a_33 * a_45 * a_54
            + a_02 * a_11 * a_20 * a_34 * a_43 * a_55
            - a_02 * a_11 * a_20 * a_34 * a_45 * a_53
            - a_02 * a_11 * a_20 * a_35 * a_43 * a_54
            + a_02 * a_11 * a_20 * a_35 * a_44 * a_53
            + a_02 * a_11 * a_23 * a_30 * a_44 * a_55
            - a_02 * a_11 * a_23 * a_30 * a_45 * a_54
            - a_02 * a_11 * a_23 * a_34 * a_40 * a_55
            + a_02 * a_11 * a_23 * a_34 * a_45 * a_50
            + a_02 * a_11 * a_23 * a_35 * a_40 * a_54
            - a_02 * a_11 * a_23 * a_35 * a_44 * a_50
            - a_02 * a_11 * a_24 * a_30 * a_43 * a_55
            + a_02 * a_11 * a_24 * a_30 * a_45 * a_53
            + a_02 * a_11 * a_24 * a_33 * a_40 * a_55
            - a_02 * a_11 * a_24 * a_33 * a_45 * a_50
            - a_02 * a_11 * a_24 * a_35 * a_40 * a_53
            + a_02 * a_11 * a_24 * a_35 * a_43 * a_50
            + a_02 * a_11 * a_25 * a_30 * a_43 * a_54
            - a_02 * a_11 * a_25 * a_30 * a_44 * a_53
            - a_02 * a_11 * a_25 * a_33 * a_40 * a_54
            + a_02 * a_11 * a_25 * a_33 * a_44 * a_50
            + a_02 * a_11 * a_25 * a_34 * a_40 * a_53
            - a_02 * a_11 * a_25 * a_34 * a_43 * a_50
            + a_02 * a_13 * a_20 * a_31 * a_44 * a_55
            - a_02 * a_13 * a_20 * a_31 * a_45 * a_54
            - a_02 * a_13 * a_20 * a_34 * a_41 * a_55
            + a_02 * a_13 * a_20 * a_34 * a_45 * a_51
            + a_02 * a_13 * a_20 * a_35 * a_41 * a_54
            - a_02 * a_13 * a_20 * a_35 * a_44 * a_51
            - a_02 * a_13 * a_21 * a_30 * a_44 * a_55
            + a_02 * a_13 * a_21 * a_30 * a_45 * a_54
            + a_02 * a_13 * a_21 * a_34 * a_40 * a_55
            - a_02 * a_13 * a_21 * a_34 * a_45 * a_50
            - a_02 * a_13 * a_21 * a_35 * a_40 * a_54
            + a_02 * a_13 * a_21 * a_35 * a_44 * a_50
            + a_02 * a_13 * a_24 * a_30 * a_41 * a_55
            - a_02 * a_13 * a_24 * a_30 * a_45 * a_51
            - a_02 * a_13 * a_24 * a_31 * a_40 * a_55
            + a_02 * a_13 * a_24 * a_31 * a_45 * a_50
            + a_02 * a_13 * a_24 * a_35 * a_40 * a_51
            - a_02 * a_13 * a_24 * a_35 * a_41 * a_50
            - a_02 * a_13 * a_25 * a_30 * a_41 * a_54
            + a_02 * a_13 * a_25 * a_30 * a_44 * a_51
            + a_02 * a_13 * a_25 * a_31 * a_40 * a_54
            - a_02 * a_13 * a_25 * a_31 * a_44 * a_50
            - a_02 * a_13 * a_25 * a_34 * a_40 * a_51
            + a_02 * a_13 * a_25 * a_34 * a_41 * a_50
            - a_02 * a_14 * a_20 * a_31 * a_43 * a_55
            + a_02 * a_14 * a_20 * a_31 * a_45 * a_53
            + a_02 * a_14 * a_20 * a_33 * a_41 * a_55
            - a_02 * a_14 * a_20 * a_33 * a_45 * a_51
            - a_02 * a_14 * a_20 * a_35 * a_41 * a_53
            + a_02 * a_14 * a_20 * a_35 * a_43 * a_51
            + a_02 * a_14 * a_21 * a_30 * a_43 * a_55
            - a_02 * a_14 * a_21 * a_30 * a_45 * a_53
            - a_02 * a_14 * a_21 * a_33 * a_40 * a_55
            + a_02 * a_14 * a_21 * a_33 * a_45 * a_50
            + a_02 * a_14 * a_21 * a_35 * a_40 * a_53
            - a_02 * a_14 * a_21 * a_35 * a_43 * a_50
            - a_02 * a_14 * a_23 * a_30 * a_41 * a_55
            + a_02 * a_14 * a_23 * a_30 * a_45 * a_51
            + a_02 * a_14 * a_23 * a_31 * a_40 * a_55
            - a_02 * a_14 * a_23 * a_31 * a_45 * a_50
            - a_02 * a_14 * a_23 * a_35 * a_40 * a_51
            + a_02 * a_14 * a_23 * a_35 * a_41 * a_50
            + a_02 * a_14 * a_25 * a_30 * a_41 * a_53
            - a_02 * a_14 * a_25 * a_30 * a_43 * a_51
            - a_02 * a_14 * a_25 * a_31 * a_40 * a_53
            + a_02 * a_14 * a_25 * a_31 * a_43 * a_50
            + a_02 * a_14 * a_25 * a_33 * a_40 * a_51
            - a_02 * a_14 * a_25 * a_33 * a_41 * a_50
            + a_02 * a_15 * a_20 * a_31 * a_43 * a_54
            - a_02 * a_15 * a_20 * a_31 * a_44 * a_53
            - a_02 * a_15 * a_20 * a_33 * a_41 * a_54
            + a_02 * a_15 * a_20 * a_33 * a_44 * a_51
            + a_02 * a_15 * a_20 * a_34 * a_41 * a_53
            - a_02 * a_15 * a_20 * a_34 * a_43 * a_51
            - a_02 * a_15 * a_21 * a_30 * a_43 * a_54
            + a_02 * a_15 * a_21 * a_30 * a_44 * a_53
            + a_02 * a_15 * a_21 * a_33 * a_40 * a_54
            - a_02 * a_15 * a_21 * a_33 * a_44 * a_50
            - a_02 * a_15 * a_21 * a_34 * a_40 * a_53
            + a_02 * a_15 * a_21 * a_34 * a_43 * a_50
            + a_02 * a_15 * a_23 * a_30 * a_41 * a_54
            - a_02 * a_15 * a_23 * a_30 * a_44 * a_51
            - a_02 * a_15 * a_23 * a_31 * a_40 * a_54
            + a_02 * a_15 * a_23 * a_31 * a_44 * a_50
            + a_02 * a_15 * a_23 * a_34 * a_40 * a_51
            - a_02 * a_15 * a_23 * a_34 * a_41 * a_50
            - a_02 * a_15 * a_24 * a_30 * a_41 * a_53
            + a_02 * a_15 * a_24 * a_30 * a_43 * a_51
            + a_02 * a_15 * a_24 * a_31 * a_40 * a_53
            - a_02 * a_15 * a_24 * a_31 * a_43 * a_50
            - a_02 * a_15 * a_24 * a_33 * a_40 * a_51
            + a_02 * a_15 * a_24 * a_33 * a_41 * a_50
            - a_03 * a_10 * a_21 * a_32 * a_44 * a_55
            + a_03 * a_10 * a_21 * a_32 * a_45 * a_54
            + a_03 * a_10 * a_21 * a_34 * a_42 * a_55
            - a_03 * a_10 * a_21 * a_34 * a_45 * a_52
            - a_03 * a_10 * a_21 * a_35 * a_42 * a_54
            + a_03 * a_10 * a_21 * a_35 * a_44 * a_52
            + a_03 * a_10 * a_22 * a_31 * a_44 * a_55
            - a_03 * a_10 * a_22 * a_31 * a_45 * a_54
            - a_03 * a_10 * a_22 * a_34 * a_41 * a_55
            + a_03 * a_10 * a_22 * a_34 * a_45 * a_51
            + a_03 * a_10 * a_22 * a_35 * a_41 * a_54
            - a_03 * a_10 * a_22 * a_35 * a_44 * a_51
            - a_03 * a_10 * a_24 * a_31 * a_42 * a_55
            + a_03 * a_10 * a_24 * a_31 * a_45 * a_52
            + a_03 * a_10 * a_24 * a_32 * a_41 * a_55
            - a_03 * a_10 * a_24 * a_32 * a_45 * a_51
            - a_03 * a_10 * a_24 * a_35 * a_41 * a_52
            + a_03 * a_10 * a_24 * a_35 * a_42 * a_51
            + a_03 * a_10 * a_25 * a_31 * a_42 * a_54
            - a_03 * a_10 * a_25 * a_31 * a_44 * a_52
            - a_03 * a_10 * a_25 * a_32 * a_41 * a_54
            + a_03 * a_10 * a_25 * a_32 * a_44 * a_51
            + a_03 * a_10 * a_25 * a_34 * a_41 * a_52
            - a_03 * a_10 * a_25 * a_34 * a_42 * a_51
            + a_03 * a_11 * a_20 * a_32 * a_44 * a_55
            - a_03 * a_11 * a_20 * a_32 * a_45 * a_54
            - a_03 * a_11 * a_20 * a_34 * a_42 * a_55
            + a_03 * a_11 * a_20 * a_34 * a_45 * a_52
            + a_03 * a_11 * a_20 * a_35 * a_42 * a_54
            - a_03 * a_11 * a_20 * a_35 * a_44 * a_52
            - a_03 * a_11 * a_22 * a_30 * a_44 * a_55
            + a_03 * a_11 * a_22 * a_30 * a_45 * a_54
            + a_03 * a_11 * a_22 * a_34 * a_40 * a_55
            - a_03 * a_11 * a_22 * a_34 * a_45 * a_50
            - a_03 * a_11 * a_22 * a_35 * a_40 * a_54
            + a_03 * a_11 * a_22 * a_35 * a_44 * a_50
            + a_03 * a_11 * a_24 * a_30 * a_42 * a_55
            - a_03 * a_11 * a_24 * a_30 * a_45 * a_52
            - a_03 * a_11 * a_24 * a_32 * a_40 * a_55
            + a_03 * a_11 * a_24 * a_32 * a_45 * a_50
            + a_03 * a_11 * a_24 * a_35 * a_40 * a_52
            - a_03 * a_11 * a_24 * a_35 * a_42 * a_50
            - a_03 * a_11 * a_25 * a_30 * a_42 * a_54
            + a_03 * a_11 * a_25 * a_30 * a_44 * a_52
            + a_03 * a_11 * a_25 * a_32 * a_40 * a_54
            - a_03 * a_11 * a_25 * a_32 * a_44 * a_50
            - a_03 * a_11 * a_25 * a_34 * a_40 * a_52
            + a_03 * a_11 * a_25 * a_34 * a_42 * a_50
            - a_03 * a_12 * a_20 * a_31 * a_44 * a_55
            + a_03 * a_12 * a_20 * a_31 * a_45 * a_54
            + a_03 * a_12 * a_20 * a_34 * a_41 * a_55
            - a_03 * a_12 * a_20 * a_34 * a_45 * a_51
            - a_03 * a_12 * a_20 * a_35 * a_41 * a_54
            + a_03 * a_12 * a_20 * a_35 * a_44 * a_51
            + a_03 * a_12 * a_21 * a_30 * a_44 * a_55
            - a_03 * a_12 * a_21 * a_30 * a_45 * a_54
            - a_03 * a_12 * a_21 * a_34 * a_40 * a_55
            + a_03 * a_12 * a_21 * a_34 * a_45 * a_50
            + a_03 * a_12 * a_21 * a_35 * a_40 * a_54
            - a_03 * a_12 * a_21 * a_35 * a_44 * a_50
            - a_03 * a_12 * a_24 * a_30 * a_41 * a_55
            + a_03 * a_12 * a_24 * a_30 * a_45 * a_51
            + a_03 * a_12 * a_24 * a_31 * a_40 * a_55
            - a_03 * a_12 * a_24 * a_31 * a_45 * a_50
            - a_03 * a_12 * a_24 * a_35 * a_40 * a_51
            + a_03 * a_12 * a_24 * a_35 * a_41 * a_50
            + a_03 * a_12 * a_25 * a_30 * a_41 * a_54
            - a_03 * a_12 * a_25 * a_30 * a_44 * a_51
            - a_03 * a_12 * a_25 * a_31 * a_40 * a_54
            + a_03 * a_12 * a_25 * a_31 * a_44 * a_50
            + a_03 * a_12 * a_25 * a_34 * a_40 * a_51
            - a_03 * a_12 * a_25 * a_34 * a_41 * a_50
            + a_03 * a_14 * a_20 * a_31 * a_42 * a_55
            - a_03 * a_14 * a_20 * a_31 * a_45 * a_52
            - a_03 * a_14 * a_20 * a_32 * a_41 * a_55
            + a_03 * a_14 * a_20 * a_32 * a_45 * a_51
            + a_03 * a_14 * a_20 * a_35 * a_41 * a_52
            - a_03 * a_14 * a_20 * a_35 * a_42 * a_51
            - a_03 * a_14 * a_21 * a_30 * a_42 * a_55
            + a_03 * a_14 * a_21 * a_30 * a_45 * a_52
            + a_03 * a_14 * a_21 * a_32 * a_40 * a_55
            - a_03 * a_14 * a_21 * a_32 * a_45 * a_50
            - a_03 * a_14 * a_21 * a_35 * a_40 * a_52
            + a_03 * a_14 * a_21 * a_35 * a_42 * a_50
            + a_03 * a_14 * a_22 * a_30 * a_41 * a_55
            - a_03 * a_14 * a_22 * a_30 * a_45 * a_51
            - a_03 * a_14 * a_22 * a_31 * a_40 * a_55
            + a_03 * a_14 * a_22 * a_31 * a_45 * a_50
            + a_03 * a_14 * a_22 * a_35 * a_40 * a_51
            - a_03 * a_14 * a_22 * a_35 * a_41 * a_50
            - a_03 * a_14 * a_25 * a_30 * a_41 * a_52
            + a_03 * a_14 * a_25 * a_30 * a_42 * a_51
            + a_03 * a_14 * a_25 * a_31 * a_40 * a_52
            - a_03 * a_14 * a_25 * a_31 * a_42 * a_50
            - a_03 * a_14 * a_25 * a_32 * a_40 * a_51
            + a_03 * a_14 * a_25 * a_32 * a_41 * a_50
            - a_03 * a_15 * a_20 * a_31 * a_42 * a_54
            + a_03 * a_15 * a_20 * a_31 * a_44 * a_52
            + a_03 * a_15 * a_20 * a_32 * a_41 * a_54
            - a_03 * a_15 * a_20 * a_32 * a_44 * a_51
            - a_03 * a_15 * a_20 * a_34 * a_41 * a_52
            + a_03 * a_15 * a_20 * a_34 * a_42 * a_51
            + a_03 * a_15 * a_21 * a_30 * a_42 * a_54
            - a_03 * a_15 * a_21 * a_30 * a_44 * a_52
            - a_03 * a_15 * a_21 * a_32 * a_40 * a_54
            + a_03 * a_15 * a_21 * a_32 * a_44 * a_50
            + a_03 * a_15 * a_21 * a_34 * a_40 * a_52
            - a_03 * a_15 * a_21 * a_34 * a_42 * a_50
            - a_03 * a_15 * a_22 * a_30 * a_41 * a_54
            + a_03 * a_15 * a_22 * a_30 * a_44 * a_51
            + a_03 * a_15 * a_22 * a_31 * a_40 * a_54
            - a_03 * a_15 * a_22 * a_31 * a_44 * a_50
            - a_03 * a_15 * a_22 * a_34 * a_40 * a_51
            + a_03 * a_15 * a_22 * a_34 * a_41 * a_50
            + a_03 * a_15 * a_24 * a_30 * a_41 * a_52
            - a_03 * a_15 * a_24 * a_30 * a_42 * a_51
            - a_03 * a_15 * a_24 * a_31 * a_40 * a_52
            + a_03 * a_15 * a_24 * a_31 * a_42 * a_50
            + a_03 * a_15 * a_24 * a_32 * a_40 * a_51
            - a_03 * a_15 * a_24 * a_32 * a_41 * a_50
            + a_04 * a_10 * a_21 * a_32 * a_43 * a_55
            - a_04 * a_10 * a_21 * a_32 * a_45 * a_53
            - a_04 * a_10 * a_21 * a_33 * a_42 * a_55
            + a_04 * a_10 * a_21 * a_33 * a_45 * a_52
            + a_04 * a_10 * a_21 * a_35 * a_42 * a_53
            - a_04 * a_10 * a_21 * a_35 * a_43 * a_52
            - a_04 * a_10 * a_22 * a_31 * a_43 * a_55
            + a_04 * a_10 * a_22 * a_31 * a_45 * a_53
            + a_04 * a_10 * a_22 * a_33 * a_41 * a_55
            - a_04 * a_10 * a_22 * a_33 * a_45 * a_51
            - a_04 * a_10 * a_22 * a_35 * a_41 * a_53
            + a_04 * a_10 * a_22 * a_35 * a_43 * a_51
            + a_04 * a_10 * a_23 * a_31 * a_42 * a_55
            - a_04 * a_10 * a_23 * a_31 * a_45 * a_52
            - a_04 * a_10 * a_23 * a_32 * a_41 * a_55
            + a_04 * a_10 * a_23 * a_32 * a_45 * a_51
            + a_04 * a_10 * a_23 * a_35 * a_41 * a_52
            - a_04 * a_10 * a_23 * a_35 * a_42 * a_51
            - a_04 * a_10 * a_25 * a_31 * a_42 * a_53
            + a_04 * a_10 * a_25 * a_31 * a_43 * a_52
            + a_04 * a_10 * a_25 * a_32 * a_41 * a_53
            - a_04 * a_10 * a_25 * a_32 * a_43 * a_51
            - a_04 * a_10 * a_25 * a_33 * a_41 * a_52
            + a_04 * a_10 * a_25 * a_33 * a_42 * a_51
            - a_04 * a_11 * a_20 * a_32 * a_43 * a_55
            + a_04 * a_11 * a_20 * a_32 * a_45 * a_53
            + a_04 * a_11 * a_20 * a_33 * a_42 * a_55
            - a_04 * a_11 * a_20 * a_33 * a_45 * a_52
            - a_04 * a_11 * a_20 * a_35 * a_42 * a_53
            + a_04 * a_11 * a_20 * a_35 * a_43 * a_52
            + a_04 * a_11 * a_22 * a_30 * a_43 * a_55
            - a_04 * a_11 * a_22 * a_30 * a_45 * a_53
            - a_04 * a_11 * a_22 * a_33 * a_40 * a_55
            + a_04 * a_11 * a_22 * a_33 * a_45 * a_50
            + a_04 * a_11 * a_22 * a_35 * a_40 * a_53
            - a_04 * a_11 * a_22 * a_35 * a_43 * a_50
            - a_04 * a_11 * a_23 * a_30 * a_42 * a_55
            + a_04 * a_11 * a_23 * a_30 * a_45 * a_52
            + a_04 * a_11 * a_23 * a_32 * a_40 * a_55
            - a_04 * a_11 * a_23 * a_32 * a_45 * a_50
            - a_04 * a_11 * a_23 * a_35 * a_40 * a_52
            + a_04 * a_11 * a_23 * a_35 * a_42 * a_50
            + a_04 * a_11 * a_25 * a_30 * a_42 * a_53
            - a_04 * a_11 * a_25 * a_30 * a_43 * a_52
            - a_04 * a_11 * a_25 * a_32 * a_40 * a_53
            + a_04 * a_11 * a_25 * a_32 * a_43 * a_50
            + a_04 * a_11 * a_25 * a_33 * a_40 * a_52
            - a_04 * a_11 * a_25 * a_33 * a_42 * a_50
            + a_04 * a_12 * a_20 * a_31 * a_43 * a_55
            - a_04 * a_12 * a_20 * a_31 * a_45 * a_53
            - a_04 * a_12 * a_20 * a_33 * a_41 * a_55
            + a_04 * a_12 * a_20 * a_33 * a_45 * a_51
            + a_04 * a_12 * a_20 * a_35 * a_41 * a_53
            - a_04 * a_12 * a_20 * a_35 * a_43 * a_51
            - a_04 * a_12 * a_21 * a_30 * a_43 * a_55
            + a_04 * a_12 * a_21 * a_30 * a_45 * a_53
            + a_04 * a_12 * a_21 * a_33 * a_40 * a_55
            - a_04 * a_12 * a_21 * a_33 * a_45 * a_50
            - a_04 * a_12 * a_21 * a_35 * a_40 * a_53
            + a_04 * a_12 * a_21 * a_35 * a_43 * a_50
            + a_04 * a_12 * a_23 * a_30 * a_41 * a_55
            - a_04 * a_12 * a_23 * a_30 * a_45 * a_51
            - a_04 * a_12 * a_23 * a_31 * a_40 * a_55
            + a_04 * a_12 * a_23 * a_31 * a_45 * a_50
            + a_04 * a_12 * a_23 * a_35 * a_40 * a_51
            - a_04 * a_12 * a_23 * a_35 * a_41 * a_50
            - a_04 * a_12 * a_25 * a_30 * a_41 * a_53
            + a_04 * a_12 * a_25 * a_30 * a_43 * a_51
            + a_04 * a_12 * a_25 * a_31 * a_40 * a_53
            - a_04 * a_12 * a_25 * a_31 * a_43 * a_50
            - a_04 * a_12 * a_25 * a_33 * a_40 * a_51
            + a_04 * a_12 * a_25 * a_33 * a_41 * a_50
            - a_04 * a_13 * a_20 * a_31 * a_42 * a_55
            + a_04 * a_13 * a_20 * a_31 * a_45 * a_52
            + a_04 * a_13 * a_20 * a_32 * a_41 * a_55
            - a_04 * a_13 * a_20 * a_32 * a_45 * a_51
            - a_04 * a_13 * a_20 * a_35 * a_41 * a_52
            + a_04 * a_13 * a_20 * a_35 * a_42 * a_51
            + a_04 * a_13 * a_21 * a_30 * a_42 * a_55
            - a_04 * a_13 * a_21 * a_30 * a_45 * a_52
            - a_04 * a_13 * a_21 * a_32 * a_40 * a_55
            + a_04 * a_13 * a_21 * a_32 * a_45 * a_50
            + a_04 * a_13 * a_21 * a_35 * a_40 * a_52
            - a_04 * a_13 * a_21 * a_35 * a_42 * a_50
            - a_04 * a_13 * a_22 * a_30 * a_41 * a_55
            + a_04 * a_13 * a_22 * a_30 * a_45 * a_51
            + a_04 * a_13 * a_22 * a_31 * a_40 * a_55
            - a_04 * a_13 * a_22 * a_31 * a_45 * a_50
            - a_04 * a_13 * a_22 * a_35 * a_40 * a_51
            + a_04 * a_13 * a_22 * a_35 * a_41 * a_50
            + a_04 * a_13 * a_25 * a_30 * a_41 * a_52
            - a_04 * a_13 * a_25 * a_30 * a_42 * a_51
            - a_04 * a_13 * a_25 * a_31 * a_40 * a_52
            + a_04 * a_13 * a_25 * a_31 * a_42 * a_50
            + a_04 * a_13 * a_25 * a_32 * a_40 * a_51
            - a_04 * a_13 * a_25 * a_32 * a_41 * a_50
            + a_04 * a_15 * a_20 * a_31 * a_42 * a_53
            - a_04 * a_15 * a_20 * a_31 * a_43 * a_52
            - a_04 * a_15 * a_20 * a_32 * a_41 * a_53
            + a_04 * a_15 * a_20 * a_32 * a_43 * a_51
            + a_04 * a_15 * a_20 * a_33 * a_41 * a_52
            - a_04 * a_15 * a_20 * a_33 * a_42 * a_51
            - a_04 * a_15 * a_21 * a_30 * a_42 * a_53
            + a_04 * a_15 * a_21 * a_30 * a_43 * a_52
            + a_04 * a_15 * a_21 * a_32 * a_40 * a_53
            - a_04 * a_15 * a_21 * a_32 * a_43 * a_50
            - a_04 * a_15 * a_21 * a_33 * a_40 * a_52
            + a_04 * a_15 * a_21 * a_33 * a_42 * a_50
            + a_04 * a_15 * a_22 * a_30 * a_41 * a_53
            - a_04 * a_15 * a_22 * a_30 * a_43 * a_51
            - a_04 * a_15 * a_22 * a_31 * a_40 * a_53
            + a_04 * a_15 * a_22 * a_31 * a_43 * a_50
            + a_04 * a_15 * a_22 * a_33 * a_40 * a_51
            - a_04 * a_15 * a_22 * a_33 * a_41 * a_50
            - a_04 * a_15 * a_23 * a_30 * a_41 * a_52
            + a_04 * a_15 * a_23 * a_30 * a_42 * a_51
            + a_04 * a_15 * a_23 * a_31 * a_40 * a_52
            - a_04 * a_15 * a_23 * a_31 * a_42 * a_50
            - a_04 * a_15 * a_23 * a_32 * a_40 * a_51
            + a_04 * a_15 * a_23 * a_32 * a_41 * a_50
            - a_05 * a_10 * a_21 * a_32 * a_43 * a_54
            + a_05 * a_10 * a_21 * a_32 * a_44 * a_53
            + a_05 * a_10 * a_21 * a_33 * a_42 * a_54
            - a_05 * a_10 * a_21 * a_33 * a_44 * a_52
            - a_05 * a_10 * a_21 * a_34 * a_42 * a_53
            + a_05 * a_10 * a_21 * a_34 * a_43 * a_52
            + a_05 * a_10 * a_22 * a_31 * a_43 * a_54
            - a_05 * a_10 * a_22 * a_31 * a_44 * a_53
            - a_05 * a_10 * a_22 * a_33 * a_41 * a_54
            + a_05 * a_10 * a_22 * a_33 * a_44 * a_51
            + a_05 * a_10 * a_22 * a_34 * a_41 * a_53
            - a_05 * a_10 * a_22 * a_34 * a_43 * a_51
            - a_05 * a_10 * a_23 * a_31 * a_42 * a_54
            + a_05 * a_10 * a_23 * a_31 * a_44 * a_52
            + a_05 * a_10 * a_23 * a_32 * a_41 * a_54
            - a_05 * a_10 * a_23 * a_32 * a_44 * a_51
            - a_05 * a_10 * a_23 * a_34 * a_41 * a_52
            + a_05 * a_10 * a_23 * a_34 * a_42 * a_51
            + a_05 * a_10 * a_24 * a_31 * a_42 * a_53
            - a_05 * a_10 * a_24 * a_31 * a_43 * a_52
            - a_05 * a_10 * a_24 * a_32 * a_41 * a_53
            + a_05 * a_10 * a_24 * a_32 * a_43 * a_51
            + a_05 * a_10 * a_24 * a_33 * a_41 * a_52
            - a_05 * a_10 * a_24 * a_33 * a_42 * a_51
            + a_05 * a_11 * a_20 * a_32 * a_43 * a_54
            - a_05 * a_11 * a_20 * a_32 * a_44 * a_53
            - a_05 * a_11 * a_20 * a_33 * a_42 * a_54
            + a_05 * a_11 * a_20 * a_33 * a_44 * a_52
            + a_05 * a_11 * a_20 * a_34 * a_42 * a_53
            - a_05 * a_11 * a_20 * a_34 * a_43 * a_52
            - a_05 * a_11 * a_22 * a_30 * a_43 * a_54
            + a_05 * a_11 * a_22 * a_30 * a_44 * a_53
            + a_05 * a_11 * a_22 * a_33 * a_40 * a_54
            - a_05 * a_11 * a_22 * a_33 * a_44 * a_50
            - a_05 * a_11 * a_22 * a_34 * a_40 * a_53
            + a_05 * a_11 * a_22 * a_34 * a_43 * a_50
            + a_05 * a_11 * a_23 * a_30 * a_42 * a_54
            - a_05 * a_11 * a_23 * a_30 * a_44 * a_52
            - a_05 * a_11 * a_23 * a_32 * a_40 * a_54
            + a_05 * a_11 * a_23 * a_32 * a_44 * a_50
            + a_05 * a_11 * a_23 * a_34 * a_40 * a_52
            - a_05 * a_11 * a_23 * a_34 * a_42 * a_50
            - a_05 * a_11 * a_24 * a_30 * a_42 * a_53
            + a_05 * a_11 * a_24 * a_30 * a_43 * a_52
            + a_05 * a_11 * a_24 * a_32 * a_40 * a_53
            - a_05 * a_11 * a_24 * a_32 * a_43 * a_50
            - a_05 * a_11 * a_24 * a_33 * a_40 * a_52
            + a_05 * a_11 * a_24 * a_33 * a_42 * a_50
            - a_05 * a_12 * a_20 * a_31 * a_43 * a_54
            + a_05 * a_12 * a_20 * a_31 * a_44 * a_53
            + a_05 * a_12 * a_20 * a_33 * a_41 * a_54
            - a_05 * a_12 * a_20 * a_33 * a_44 * a_51
            - a_05 * a_12 * a_20 * a_34 * a_41 * a_53
            + a_05 * a_12 * a_20 * a_34 * a_43 * a_51
            + a_05 * a_12 * a_21 * a_30 * a_43 * a_54
            - a_05 * a_12 * a_21 * a_30 * a_44 * a_53
            - a_05 * a_12 * a_21 * a_33 * a_40 * a_54
            + a_05 * a_12 * a_21 * a_33 * a_44 * a_50
            + a_05 * a_12 * a_21 * a_34 * a_40 * a_53
            - a_05 * a_12 * a_21 * a_34 * a_43 * a_50
            - a_05 * a_12 * a_23 * a_30 * a_41 * a_54
            + a_05 * a_12 * a_23 * a_30 * a_44 * a_51
            + a_05 * a_12 * a_23 * a_31 * a_40 * a_54
            - a_05 * a_12 * a_23 * a_31 * a_44 * a_50
            - a_05 * a_12 * a_23 * a_34 * a_40 * a_51
            + a_05 * a_12 * a_23 * a_34 * a_41 * a_50
            + a_05 * a_12 * a_24 * a_30 * a_41 * a_53
            - a_05 * a_12 * a_24 * a_30 * a_43 * a_51
            - a_05 * a_12 * a_24 * a_31 * a_40 * a_53
            + a_05 * a_12 * a_24 * a_31 * a_43 * a_50
            + a_05 * a_12 * a_24 * a_33 * a_40 * a_51
            - a_05 * a_12 * a_24 * a_33 * a_41 * a_50
            + a_05 * a_13 * a_20 * a_31 * a_42 * a_54
            - a_05 * a_13 * a_20 * a_31 * a_44 * a_52
            - a_05 * a_13 * a_20 * a_32 * a_41 * a_54
            + a_05 * a_13 * a_20 * a_32 * a_44 * a_51
            + a_05 * a_13 * a_20 * a_34 * a_41 * a_52
            - a_05 * a_13 * a_20 * a_34 * a_42 * a_51
            - a_05 * a_13 * a_21 * a_30 * a_42 * a_54
            + a_05 * a_13 * a_21 * a_30 * a_44 * a_52
            + a_05 * a_13 * a_21 * a_32 * a_40 * a_54
            - a_05 * a_13 * a_21 * a_32 * a_44 * a_50
            - a_05 * a_13 * a_21 * a_34 * a_40 * a_52
            + a_05 * a_13 * a_21 * a_34 * a_42 * a_50
            + a_05 * a_13 * a_22 * a_30 * a_41 * a_54
            - a_05 * a_13 * a_22 * a_30 * a_44 * a_51
            - a_05 * a_13 * a_22 * a_31 * a_40 * a_54
            + a_05 * a_13 * a_22 * a_31 * a_44 * a_50
            + a_05 * a_13 * a_22 * a_34 * a_40 * a_51
            - a_05 * a_13 * a_22 * a_34 * a_41 * a_50
            - a_05 * a_13 * a_24 * a_30 * a_41 * a_52
            + a_05 * a_13 * a_24 * a_30 * a_42 * a_51
            + a_05 * a_13 * a_24 * a_31 * a_40 * a_52
            - a_05 * a_13 * a_24 * a_31 * a_42 * a_50
            - a_05 * a_13 * a_24 * a_32 * a_40 * a_51
            + a_05 * a_13 * a_24 * a_32 * a_41 * a_50
            - a_05 * a_14 * a_20 * a_31 * a_42 * a_53
            + a_05 * a_14 * a_20 * a_31 * a_43 * a_52
            + a_05 * a_14 * a_20 * a_32 * a_41 * a_53
            - a_05 * a_14 * a_20 * a_32 * a_43 * a_51
            - a_05 * a_14 * a_20 * a_33 * a_41 * a_52
            + a_05 * a_14 * a_20 * a_33 * a_42 * a_51
            + a_05 * a_14 * a_21 * a_30 * a_42 * a_53
            - a_05 * a_14 * a_21 * a_30 * a_43 * a_52
            - a_05 * a_14 * a_21 * a_32 * a_40 * a_53
            + a_05 * a_14 * a_21 * a_32 * a_43 * a_50
            + a_05 * a_14 * a_21 * a_33 * a_40 * a_52
            - a_05 * a_14 * a_21 * a_33 * a_42 * a_50
            - a_05 * a_14 * a_22 * a_30 * a_41 * a_53
            + a_05 * a_14 * a_22 * a_30 * a_43 * a_51
            + a_05 * a_14 * a_22 * a_31 * a_40 * a_53
            - a_05 * a_14 * a_22 * a_31 * a_43 * a_50
            - a_05 * a_14 * a_22 * a_33 * a_40 * a_51
            + a_05 * a_14 * a_22 * a_33 * a_41 * a_50
            + a_05 * a_14 * a_23 * a_30 * a_41 * a_52
            - a_05 * a_14 * a_23 * a_30 * a_42 * a_51
            - a_05 * a_14 * a_23 * a_31 * a_40 * a_52
            + a_05 * a_14 * a_23 * a_31 * a_42 * a_50
            + a_05 * a_14 * a_23 * a_32 * a_40 * a_51
            - a_05 * a_14 * a_23 * a_32 * a_41 * a_50
    }
    ///
    /// Calculate the inverse of the M66 via tha Adjoint Matrix:
    ///
    /// A^(-1) = (1/det) * Adj
    ///
    /// where: Adj = Cofactor.Transpose()
    ///
    /// Cofactor = (-1)^(i+j) * M(i, j).det()
    #[inline(always)]
    fn inverse(&self) -> Option<Self> {
        let one = T::one();
        let det = self.det();
        if !nearly_zero(det) {
            let mut cofactors: M66<T> = M66::zeros();
            for i in 0..6 {
                for j in 0..6 {
                    let sign = if (i + j) % 2 == 0 {one} else {-one};
                    // transpose in place
                    cofactors[(j, i)] = sign * self.get_submatrix((i, j)).det();
                }
            }
            Some(cofactors / det)
        } else {
            None
        }
    }

    /// Calculate de QR factorization of the M66 via gram-schmidt
    /// orthogonalization process
    fn qr(&self) -> Option<(Self, Self)> {
        if !nearly_zero(self.det()) {
            let cols = self.get_cols();
            let mut q: [V6<T>; 6] = *M66::zeros().get_cols();
            for i in 0..q.len() {
                let mut q_tilde = cols[i];
                for k in 0..i {
                    q_tilde -= q[k] * project_x_over_y(&*cols[i], &*q[k]);
                }
                normalize(&mut *q_tilde);
                q[i] = q_tilde;
            }
            // TODO(elsuizo:2020-08-05): do this with a another for loop
            let basis = V6::new([q[0], q[1], q[2], q[3], q[4], q[5]]);
            let q     = M66::new_from_vecs(basis);
            let r     = q.transpose() * (*self);
            Some((q, r))
        } else {
            None
        }
    }

}

// TODO(elsuizo:2020-08-29): check with this in the future!!!
// impl<T: Float> M66<T> {
//
//     const fn get_sign(i: usize, j: usize) -> T {
//         let one = T::one();
//
//         let m = m66_new!(one,-one, one,-one, one,-one;
//                         -one, one,-one, one,-one, one;
//                          one,-one, one,-one, one,-one;
//                         -one, one,-one, one,-one, one;
//                          one,-one, one,-one, one,-one;
//                         -one, one,-one, one,-one, one);
//         m[(i, j)]
//     }
//
// }


// M66 + M66
impl<T: Num + Copy> Add for M66<T> {
    type Output = Self;

    fn add(self, rhs: Self) -> Self {
        let a_00 = self[(0, 0)];
        let a_01 = self[(0, 1)];
        let a_02 = self[(0, 2)];
        let a_03 = self[(0, 3)];
        let a_04 = self[(0, 4)];
        let a_05 = self[(0, 5)];
        let a_10 = self[(1, 0)];
        let a_11 = self[(1, 1)];
        let a_12 = self[(1, 2)];
        let a_13 = self[(1, 3)];
        let a_14 = self[(1, 4)];
        let a_15 = self[(1, 5)];
        let a_20 = self[(2, 0)];
        let a_21 = self[(2, 1)];
        let a_22 = self[(2, 2)];
        let a_23 = self[(2, 3)];
        let a_24 = self[(2, 4)];
        let a_25 = self[(2, 5)];
        let a_30 = self[(3, 0)];
        let a_31 = self[(3, 1)];
        let a_32 = self[(3, 2)];
        let a_33 = self[(3, 3)];
        let a_34 = self[(3, 4)];
        let a_35 = self[(3, 5)];
        let a_40 = self[(4, 0)];
        let a_41 = self[(4, 1)];
        let a_42 = self[(4, 2)];
        let a_43 = self[(4, 3)];
        let a_44 = self[(4, 4)];
        let a_45 = self[(4, 5)];
        let a_50 = self[(5, 0)];
        let a_51 = self[(5, 1)];
        let a_52 = self[(5, 2)];
        let a_53 = self[(5, 3)];
        let a_54 = self[(5, 4)];
        let a_55 = self[(5, 5)];

        let b_00 = rhs[(0, 0)];
        let b_01 = rhs[(0, 1)];
        let b_02 = rhs[(0, 2)];
        let b_03 = rhs[(0, 3)];
        let b_04 = rhs[(0, 4)];
        let b_05 = rhs[(0, 5)];
        let b_10 = rhs[(1, 0)];
        let b_11 = rhs[(1, 1)];
        let b_12 = rhs[(1, 2)];
        let b_13 = rhs[(1, 3)];
        let b_14 = rhs[(1, 4)];
        let b_15 = rhs[(1, 5)];
        let b_20 = rhs[(2, 0)];
        let b_21 = rhs[(2, 1)];
        let b_22 = rhs[(2, 2)];
        let b_23 = rhs[(2, 3)];
        let b_24 = rhs[(2, 4)];
        let b_25 = rhs[(2, 5)];
        let b_30 = rhs[(3, 0)];
        let b_31 = rhs[(3, 1)];
        let b_32 = rhs[(3, 2)];
        let b_33 = rhs[(3, 3)];
        let b_34 = rhs[(3, 4)];
        let b_35 = rhs[(3, 5)];
        let b_40 = rhs[(4, 0)];
        let b_41 = rhs[(4, 1)];
        let b_42 = rhs[(4, 2)];
        let b_43 = rhs[(4, 3)];
        let b_44 = rhs[(4, 4)];
        let b_45 = rhs[(4, 5)];
        let b_50 = rhs[(5, 0)];
        let b_51 = rhs[(5, 1)];
        let b_52 = rhs[(5, 2)];
        let b_53 = rhs[(5, 3)];
        let b_54 = rhs[(5, 4)];
        let b_55 = rhs[(5, 5)];

        M66::new([
            [
                a_00 + b_00,
                a_01 + b_01,
                a_02 + b_02,
                a_03 + b_03,
                a_04 + b_04,
                a_05 + b_05,
            ],
            [
                a_10 + b_10,
                a_11 + b_11,
                a_12 + b_12,
                a_13 + b_13,
                a_14 + b_14,
                a_15 + b_15,
            ],
            [
                a_20 + b_20,
                a_21 + b_21,
                a_22 + b_22,
                a_23 + b_23,
                a_24 + b_24,
                a_25 + b_25,
            ],
            [
                a_30 + b_30,
                a_31 + b_31,
                a_32 + b_32,
                a_33 + b_33,
                a_34 + b_34,
                a_35 + b_35,
            ],
            [
                a_40 + b_40,
                a_41 + b_41,
                a_42 + b_42,
                a_43 + b_43,
                a_44 + b_44,
                a_45 + b_45,
            ],
            [
                a_50 + b_50,
                a_51 + b_51,
                a_52 + b_52,
                a_53 + b_53,
                a_54 + b_54,
                a_55 + b_55,
            ],
        ])
    }
}

// M66 += M66
impl<T: Num + Copy> AddAssign for M66<T> {
    fn add_assign(&mut self, other: Self) {
        *self = *self + other
    }
}

// M66 - M66
impl<T: Num + Copy> Sub for M66<T> {
    type Output = Self;

    fn sub(self, rhs: Self) -> Self {
        let a_00 = self[(0, 0)];
        let a_01 = self[(0, 1)];
        let a_02 = self[(0, 2)];
        let a_03 = self[(0, 3)];
        let a_04 = self[(0, 4)];
        let a_05 = self[(0, 5)];
        let a_10 = self[(1, 0)];
        let a_11 = self[(1, 1)];
        let a_12 = self[(1, 2)];
        let a_13 = self[(1, 3)];
        let a_14 = self[(1, 4)];
        let a_15 = self[(1, 5)];
        let a_20 = self[(2, 0)];
        let a_21 = self[(2, 1)];
        let a_22 = self[(2, 2)];
        let a_23 = self[(2, 3)];
        let a_24 = self[(2, 4)];
        let a_25 = self[(2, 5)];
        let a_30 = self[(3, 0)];
        let a_31 = self[(3, 1)];
        let a_32 = self[(3, 2)];
        let a_33 = self[(3, 3)];
        let a_34 = self[(3, 4)];
        let a_35 = self[(3, 5)];
        let a_40 = self[(4, 0)];
        let a_41 = self[(4, 1)];
        let a_42 = self[(4, 2)];
        let a_43 = self[(4, 3)];
        let a_44 = self[(4, 4)];
        let a_45 = self[(4, 5)];
        let a_50 = self[(5, 0)];
        let a_51 = self[(5, 1)];
        let a_52 = self[(5, 2)];
        let a_53 = self[(5, 3)];
        let a_54 = self[(5, 4)];
        let a_55 = self[(5, 5)];

        let b_00 = rhs[(0, 0)];
        let b_01 = rhs[(0, 1)];
        let b_02 = rhs[(0, 2)];
        let b_03 = rhs[(0, 3)];
        let b_04 = rhs[(0, 4)];
        let b_05 = rhs[(0, 5)];
        let b_10 = rhs[(1, 0)];
        let b_11 = rhs[(1, 1)];
        let b_12 = rhs[(1, 2)];
        let b_13 = rhs[(1, 3)];
        let b_14 = rhs[(1, 4)];
        let b_15 = rhs[(1, 5)];
        let b_20 = rhs[(2, 0)];
        let b_21 = rhs[(2, 1)];
        let b_22 = rhs[(2, 2)];
        let b_23 = rhs[(2, 3)];
        let b_24 = rhs[(2, 4)];
        let b_25 = rhs[(2, 5)];
        let b_30 = rhs[(3, 0)];
        let b_31 = rhs[(3, 1)];
        let b_32 = rhs[(3, 2)];
        let b_33 = rhs[(3, 3)];
        let b_34 = rhs[(3, 4)];
        let b_35 = rhs[(3, 5)];
        let b_40 = rhs[(4, 0)];
        let b_41 = rhs[(4, 1)];
        let b_42 = rhs[(4, 2)];
        let b_43 = rhs[(4, 3)];
        let b_44 = rhs[(4, 4)];
        let b_45 = rhs[(4, 5)];
        let b_50 = rhs[(5, 0)];
        let b_51 = rhs[(5, 1)];
        let b_52 = rhs[(5, 2)];
        let b_53 = rhs[(5, 3)];
        let b_54 = rhs[(5, 4)];
        let b_55 = rhs[(5, 5)];

        M66::new([
            [
                a_00 - b_00,
                a_01 - b_01,
                a_02 - b_02,
                a_03 - b_03,
                a_04 - b_04,
                a_05 - b_05,
            ],
            [
                a_10 - b_10,
                a_11 - b_11,
                a_12 - b_12,
                a_13 - b_13,
                a_14 - b_14,
                a_15 - b_15,
            ],
            [
                a_20 - b_20,
                a_21 - b_21,
                a_22 - b_22,
                a_23 - b_23,
                a_24 - b_24,
                a_25 - b_25,
            ],
            [
                a_30 - b_30,
                a_31 - b_31,
                a_32 - b_32,
                a_33 - b_33,
                a_34 - b_34,
                a_35 - b_35,
            ],
            [
                a_40 - b_40,
                a_41 - b_41,
                a_42 - b_42,
                a_43 - b_43,
                a_44 - b_44,
                a_45 - b_45,
            ],
            [
                a_50 - b_50,
                a_51 - b_51,
                a_52 - b_52,
                a_53 - b_53,
                a_54 - b_54,
                a_55 - b_55,
            ],
        ])
    }
}

// M66 -= M66
impl<T: Num + Copy> SubAssign for M66<T> {
    fn sub_assign(&mut self, other: Self) {
        *self = *self - other
    }
}

// M66 * T
impl<T: Num + Copy> Mul<T> for M66<T> {
    type Output = M66<T>;

    fn mul(self, rhs: T) -> M66<T> {
        let a_00 = self[(0, 0)] * rhs;
        let a_01 = self[(0, 1)] * rhs;
        let a_02 = self[(0, 2)] * rhs;
        let a_03 = self[(0, 3)] * rhs;
        let a_04 = self[(0, 4)] * rhs;
        let a_05 = self[(0, 5)] * rhs;
        let a_10 = self[(1, 0)] * rhs;
        let a_11 = self[(1, 1)] * rhs;
        let a_12 = self[(1, 2)] * rhs;
        let a_13 = self[(1, 3)] * rhs;
        let a_14 = self[(1, 4)] * rhs;
        let a_15 = self[(1, 5)] * rhs;
        let a_20 = self[(2, 0)] * rhs;
        let a_21 = self[(2, 1)] * rhs;
        let a_22 = self[(2, 2)] * rhs;
        let a_23 = self[(2, 3)] * rhs;
        let a_24 = self[(2, 4)] * rhs;
        let a_25 = self[(2, 5)] * rhs;
        let a_30 = self[(3, 0)] * rhs;
        let a_31 = self[(3, 1)] * rhs;
        let a_32 = self[(3, 2)] * rhs;
        let a_33 = self[(3, 3)] * rhs;
        let a_34 = self[(3, 4)] * rhs;
        let a_35 = self[(3, 5)] * rhs;
        let a_40 = self[(4, 0)] * rhs;
        let a_41 = self[(4, 1)] * rhs;
        let a_42 = self[(4, 2)] * rhs;
        let a_43 = self[(4, 3)] * rhs;
        let a_44 = self[(4, 4)] * rhs;
        let a_45 = self[(4, 5)] * rhs;
        let a_50 = self[(5, 0)] * rhs;
        let a_51 = self[(5, 1)] * rhs;
        let a_52 = self[(5, 2)] * rhs;
        let a_53 = self[(5, 3)] * rhs;
        let a_54 = self[(5, 4)] * rhs;
        let a_55 = self[(5, 5)] * rhs;

        M66::new([
            [a_00, a_01, a_02, a_03, a_04, a_05],
            [a_10, a_11, a_12, a_13, a_14, a_15],
            [a_20, a_21, a_22, a_23, a_24, a_25],
            [a_30, a_31, a_32, a_33, a_34, a_35],
            [a_40, a_41, a_42, a_43, a_44, a_45],
            [a_50, a_51, a_52, a_53, a_54, a_55],
        ])
    }
}

// M66 / constant
impl<T: Num + Copy> Div<T> for M66<T> {
    type Output = Self;

    fn div(self, rhs: T) -> Self::Output {
        let a_00 = self[(0, 0)] / rhs;
        let a_01 = self[(0, 1)] / rhs;
        let a_02 = self[(0, 2)] / rhs;
        let a_03 = self[(0, 3)] / rhs;
        let a_04 = self[(0, 4)] / rhs;
        let a_05 = self[(0, 5)] / rhs;
        let a_10 = self[(1, 0)] / rhs;
        let a_11 = self[(1, 1)] / rhs;
        let a_12 = self[(1, 2)] / rhs;
        let a_13 = self[(1, 3)] / rhs;
        let a_14 = self[(1, 4)] / rhs;
        let a_15 = self[(1, 5)] / rhs;
        let a_20 = self[(2, 0)] / rhs;
        let a_21 = self[(2, 1)] / rhs;
        let a_22 = self[(2, 2)] / rhs;
        let a_23 = self[(2, 3)] / rhs;
        let a_24 = self[(2, 4)] / rhs;
        let a_25 = self[(2, 5)] / rhs;
        let a_30 = self[(3, 0)] / rhs;
        let a_31 = self[(3, 1)] / rhs;
        let a_32 = self[(3, 2)] / rhs;
        let a_33 = self[(3, 3)] / rhs;
        let a_34 = self[(3, 4)] / rhs;
        let a_35 = self[(3, 5)] / rhs;
        let a_40 = self[(4, 0)] / rhs;
        let a_41 = self[(4, 1)] / rhs;
        let a_42 = self[(4, 2)] / rhs;
        let a_43 = self[(4, 3)] / rhs;
        let a_44 = self[(4, 4)] / rhs;
        let a_45 = self[(4, 5)] / rhs;
        let a_50 = self[(5, 0)] / rhs;
        let a_51 = self[(5, 1)] / rhs;
        let a_52 = self[(5, 2)] / rhs;
        let a_53 = self[(5, 3)] / rhs;
        let a_54 = self[(5, 4)] / rhs;
        let a_55 = self[(5, 5)] / rhs;

        M66::new([
            [a_00, a_01, a_02, a_03, a_04, a_05],
            [a_10, a_11, a_12, a_13, a_14, a_15],
            [a_20, a_21, a_22, a_23, a_24, a_25],
            [a_30, a_31, a_32, a_33, a_34, a_35],
            [a_40, a_41, a_42, a_43, a_44, a_45],
            [a_50, a_51, a_52, a_53, a_54, a_55],
        ])
    }
}

// -M66
impl<T: Num + Copy + Signed> Neg for M66<T> {
    type Output = Self;

    fn neg(self) -> Self {
        let a_00 = -self[(0, 0)];
        let a_01 = -self[(0, 1)];
        let a_02 = -self[(0, 2)];
        let a_03 = -self[(0, 3)];
        let a_04 = -self[(0, 4)];
        let a_05 = -self[(0, 5)];
        let a_10 = -self[(1, 0)];
        let a_11 = -self[(1, 1)];
        let a_12 = -self[(1, 2)];
        let a_13 = -self[(1, 3)];
        let a_14 = -self[(1, 4)];
        let a_15 = -self[(1, 5)];
        let a_20 = -self[(2, 0)];
        let a_21 = -self[(2, 1)];
        let a_22 = -self[(2, 2)];
        let a_23 = -self[(2, 3)];
        let a_24 = -self[(2, 4)];
        let a_25 = -self[(2, 5)];
        let a_30 = -self[(3, 0)];
        let a_31 = -self[(3, 1)];
        let a_32 = -self[(3, 2)];
        let a_33 = -self[(3, 3)];
        let a_34 = -self[(3, 4)];
        let a_35 = -self[(3, 5)];
        let a_40 = -self[(4, 0)];
        let a_41 = -self[(4, 1)];
        let a_42 = -self[(4, 2)];
        let a_43 = -self[(4, 3)];
        let a_44 = -self[(4, 4)];
        let a_45 = -self[(4, 5)];
        let a_50 = -self[(5, 0)];
        let a_51 = -self[(5, 1)];
        let a_52 = -self[(5, 2)];
        let a_53 = -self[(5, 3)];
        let a_54 = -self[(5, 4)];
        let a_55 = -self[(5, 5)];

    M66::new([
        [a_00, a_01, a_02, a_03, a_04, a_05],
        [a_10, a_11, a_12, a_13, a_14, a_15],
        [a_20, a_21, a_22, a_23, a_24, a_25],
        [a_30, a_31, a_32, a_33, a_34, a_35],
        [a_40, a_41, a_42, a_43, a_44, a_45],
        [a_50, a_51, a_52, a_53, a_54, a_55],
    ])
    }
}

// M66 * V6
impl<T: Num + Copy> Mul<V6<T>> for M66<T> {
    type Output = V6<T>;

    fn mul(self, rhs: V6<T>) -> V6<T> {

        let a_00 = self[(0, 0)];
        let a_01 = self[(0, 1)];
        let a_02 = self[(0, 2)];
        let a_03 = self[(0, 3)];
        let a_04 = self[(0, 4)];
        let a_05 = self[(0, 5)];
        let a_10 = self[(1, 0)];
        let a_11 = self[(1, 1)];
        let a_12 = self[(1, 2)];
        let a_13 = self[(1, 3)];
        let a_14 = self[(1, 4)];
        let a_15 = self[(1, 5)];
        let a_20 = self[(2, 0)];
        let a_21 = self[(2, 1)];
        let a_22 = self[(2, 2)];
        let a_23 = self[(2, 3)];
        let a_24 = self[(2, 4)];
        let a_25 = self[(2, 5)];
        let a_30 = self[(3, 0)];
        let a_31 = self[(3, 1)];
        let a_32 = self[(3, 2)];
        let a_33 = self[(3, 3)];
        let a_34 = self[(3, 4)];
        let a_35 = self[(3, 5)];
        let a_40 = self[(4, 0)];
        let a_41 = self[(4, 1)];
        let a_42 = self[(4, 2)];
        let a_43 = self[(4, 3)];
        let a_44 = self[(4, 4)];
        let a_45 = self[(4, 5)];
        let a_50 = self[(5, 0)];
        let a_51 = self[(5, 1)];
        let a_52 = self[(5, 2)];
        let a_53 = self[(5, 3)];
        let a_54 = self[(5, 4)];
        let a_55 = self[(5, 5)];

        let a = rhs[0];
        let b = rhs[1];
        let c = rhs[2];
        let d = rhs[3];
        let e = rhs[4];
        let f = rhs[5];

        let v0 = a_00 * a + a_01 * b + a_02 * c + a_03 * d + a_04 * e + a_05 * f;
        let v1 = a_10 * a + a_11 * b + a_12 * c + a_13 * d + a_14 * e + a_15 * f;
        let v2 = a_20 * a + a_21 * b + a_22 * c + a_23 * d + a_24 * e + a_25 * f;
        let v3 = a_30 * a + a_31 * b + a_32 * c + a_33 * d + a_34 * e + a_35 * f;
        let v4 = a_40 * a + a_41 * b + a_42 * c + a_43 * d + a_44 * e + a_45 * f;
        let v5 = a_50 * a + a_51 * b + a_52 * c + a_53 * d + a_54 * e + a_55 * f;

        V6::new([v0, v1, v2, v3, v4, v5])
    }

}

impl<T: Num + Copy> Mul for M66<T> {
    type Output = Self;

    fn mul(self, rhs: Self) -> Self {
        let a_00 = self[(0, 0)];
        let a_01 = self[(0, 1)];
        let a_02 = self[(0, 2)];
        let a_03 = self[(0, 3)];
        let a_04 = self[(0, 4)];
        let a_05 = self[(0, 5)];
        let a_10 = self[(1, 0)];
        let a_11 = self[(1, 1)];
        let a_12 = self[(1, 2)];
        let a_13 = self[(1, 3)];
        let a_14 = self[(1, 4)];
        let a_15 = self[(1, 5)];
        let a_20 = self[(2, 0)];
        let a_21 = self[(2, 1)];
        let a_22 = self[(2, 2)];
        let a_23 = self[(2, 3)];
        let a_24 = self[(2, 4)];
        let a_25 = self[(2, 5)];
        let a_30 = self[(3, 0)];
        let a_31 = self[(3, 1)];
        let a_32 = self[(3, 2)];
        let a_33 = self[(3, 3)];
        let a_34 = self[(3, 4)];
        let a_35 = self[(3, 5)];
        let a_40 = self[(4, 0)];
        let a_41 = self[(4, 1)];
        let a_42 = self[(4, 2)];
        let a_43 = self[(4, 3)];
        let a_44 = self[(4, 4)];
        let a_45 = self[(4, 5)];
        let a_50 = self[(5, 0)];
        let a_51 = self[(5, 1)];
        let a_52 = self[(5, 2)];
        let a_53 = self[(5, 3)];
        let a_54 = self[(5, 4)];
        let a_55 = self[(5, 5)];

        let b_00 = rhs[(0, 0)];
        let b_01 = rhs[(0, 1)];
        let b_02 = rhs[(0, 2)];
        let b_03 = rhs[(0, 3)];
        let b_04 = rhs[(0, 4)];
        let b_05 = rhs[(0, 5)];
        let b_10 = rhs[(1, 0)];
        let b_11 = rhs[(1, 1)];
        let b_12 = rhs[(1, 2)];
        let b_13 = rhs[(1, 3)];
        let b_14 = rhs[(1, 4)];
        let b_15 = rhs[(1, 5)];
        let b_20 = rhs[(2, 0)];
        let b_21 = rhs[(2, 1)];
        let b_22 = rhs[(2, 2)];
        let b_23 = rhs[(2, 3)];
        let b_24 = rhs[(2, 4)];
        let b_25 = rhs[(2, 5)];
        let b_30 = rhs[(3, 0)];
        let b_31 = rhs[(3, 1)];
        let b_32 = rhs[(3, 2)];
        let b_33 = rhs[(3, 3)];
        let b_34 = rhs[(3, 4)];
        let b_35 = rhs[(3, 5)];
        let b_40 = rhs[(4, 0)];
        let b_41 = rhs[(4, 1)];
        let b_42 = rhs[(4, 2)];
        let b_43 = rhs[(4, 3)];
        let b_44 = rhs[(4, 4)];
        let b_45 = rhs[(4, 5)];
        let b_50 = rhs[(5, 0)];
        let b_51 = rhs[(5, 1)];
        let b_52 = rhs[(5, 2)];
        let b_53 = rhs[(5, 3)];
        let b_54 = rhs[(5, 4)];
        let b_55 = rhs[(5, 5)];

        M66::new([
            [
                a_00 * b_00 + a_01 * b_10 + a_02 * b_20 + a_03 * b_30 + a_04 * b_40 + a_05 * b_50,
                a_00 * b_01 + a_01 * b_11 + a_02 * b_21 + a_03 * b_31 + a_04 * b_41 + a_05 * b_51,
                a_00 * b_02 + a_01 * b_12 + a_02 * b_22 + a_03 * b_32 + a_04 * b_42 + a_05 * b_52,
                a_00 * b_03 + a_01 * b_13 + a_02 * b_23 + a_03 * b_33 + a_04 * b_43 + a_05 * b_53,
                a_00 * b_04 + a_01 * b_14 + a_02 * b_24 + a_03 * b_34 + a_04 * b_44 + a_05 * b_54,
                a_00 * b_05 + a_01 * b_15 + a_02 * b_25 + a_03 * b_35 + a_04 * b_45 + a_05 * b_55,
            ],
            [
                a_10 * b_00 + a_11 * b_10 + a_12 * b_20 + a_13 * b_30 + a_14 * b_40 + a_15 * b_50,
                a_10 * b_01 + a_11 * b_11 + a_12 * b_21 + a_13 * b_31 + a_14 * b_41 + a_15 * b_51,
                a_10 * b_02 + a_11 * b_12 + a_12 * b_22 + a_13 * b_32 + a_14 * b_42 + a_15 * b_52,
                a_10 * b_03 + a_11 * b_13 + a_12 * b_23 + a_13 * b_33 + a_14 * b_43 + a_15 * b_53,
                a_10 * b_04 + a_11 * b_14 + a_12 * b_24 + a_13 * b_34 + a_14 * b_44 + a_15 * b_54,
                a_10 * b_05 + a_11 * b_15 + a_12 * b_25 + a_13 * b_35 + a_14 * b_45 + a_15 * b_55,
            ],
            [
                a_20 * b_00 + a_21 * b_10 + a_22 * b_20 + a_23 * b_30 + a_24 * b_40 + a_25 * b_50,
                a_20 * b_01 + a_21 * b_11 + a_22 * b_21 + a_23 * b_31 + a_24 * b_41 + a_25 * b_51,
                a_20 * b_02 + a_21 * b_12 + a_22 * b_22 + a_23 * b_32 + a_24 * b_42 + a_25 * b_52,
                a_20 * b_03 + a_21 * b_13 + a_22 * b_23 + a_23 * b_33 + a_24 * b_43 + a_25 * b_53,
                a_20 * b_04 + a_21 * b_14 + a_22 * b_24 + a_23 * b_34 + a_24 * b_44 + a_25 * b_54,
                a_20 * b_05 + a_21 * b_15 + a_22 * b_25 + a_23 * b_35 + a_24 * b_45 + a_25 * b_55,
            ],
            [
                a_30 * b_00 + a_31 * b_10 + a_32 * b_20 + a_33 * b_30 + a_34 * b_40 + a_35 * b_50,
                a_30 * b_01 + a_31 * b_11 + a_32 * b_21 + a_33 * b_31 + a_34 * b_41 + a_35 * b_51,
                a_30 * b_02 + a_31 * b_12 + a_32 * b_22 + a_33 * b_32 + a_34 * b_42 + a_35 * b_52,
                a_30 * b_03 + a_31 * b_13 + a_32 * b_23 + a_33 * b_33 + a_34 * b_43 + a_35 * b_53,
                a_30 * b_04 + a_31 * b_14 + a_32 * b_24 + a_33 * b_34 + a_34 * b_44 + a_35 * b_54,
                a_30 * b_05 + a_31 * b_15 + a_32 * b_25 + a_33 * b_35 + a_34 * b_45 + a_35 * b_55,
            ],
            [
                a_40 * b_00 + a_41 * b_10 + a_42 * b_20 + a_43 * b_30 + a_44 * b_40 + a_45 * b_50,
                a_40 * b_01 + a_41 * b_11 + a_42 * b_21 + a_43 * b_31 + a_44 * b_41 + a_45 * b_51,
                a_40 * b_02 + a_41 * b_12 + a_42 * b_22 + a_43 * b_32 + a_44 * b_42 + a_45 * b_52,
                a_40 * b_03 + a_41 * b_13 + a_42 * b_23 + a_43 * b_33 + a_44 * b_43 + a_45 * b_53,
                a_40 * b_04 + a_41 * b_14 + a_42 * b_24 + a_43 * b_34 + a_44 * b_44 + a_45 * b_54,
                a_40 * b_05 + a_41 * b_15 + a_42 * b_25 + a_43 * b_35 + a_44 * b_45 + a_45 * b_55,
            ],
            [
                a_50 * b_00 + a_51 * b_10 + a_52 * b_20 + a_53 * b_30 + a_54 * b_40 + a_55 * b_50,
                a_50 * b_01 + a_51 * b_11 + a_52 * b_21 + a_53 * b_31 + a_54 * b_41 + a_55 * b_51,
                a_50 * b_02 + a_51 * b_12 + a_52 * b_22 + a_53 * b_32 + a_54 * b_42 + a_55 * b_52,
                a_50 * b_03 + a_51 * b_13 + a_52 * b_23 + a_53 * b_33 + a_54 * b_43 + a_55 * b_53,
                a_50 * b_04 + a_51 * b_14 + a_52 * b_24 + a_53 * b_34 + a_54 * b_44 + a_55 * b_54,
                a_50 * b_05 + a_51 * b_15 + a_52 * b_25 + a_53 * b_35 + a_54 * b_45 + a_55 * b_55,
            ],
        ])
    }
}

impl<T: Num + Copy> M66<T> {
    /// contruct identity matrix
    pub fn identity() -> M66<T> {
        <M66<T> as One>::one()
    }

    /// construct the matrix with all zeros
    pub fn zeros() -> M66<T> {
        <M66<T> as Zero>::zero()
    }

    /// transform the matrix to a flatten vector
    pub fn as_vec(&self) -> [T; 36] {
        let mut result = [T::zero(); 36];
        for (index, element) in self.iter().flatten().enumerate() {
            result[index] = *element;
        }
        result
    }

    /// get the diagonal of the matrix
    pub fn get_diagonal(&self) -> V6<T> {
        let mut result = V6::zeros();
        let mut index: usize = 0;
        for i in 0..self.rows() {
            for j in 0..self.cols() {
                if i == j {
                    result[index] = self[(i, j)];
                    index += 1;
                }
            }
        }
        result
    }

    pub fn new_from_vecs(cols: V6<V6<T>>) -> Self {
        let mut result = Self::zeros();

        for i in 0..result.cols() {
            result[(i, 0)] = cols[0][i];
            result[(i, 1)] = cols[1][i];
            result[(i, 2)] = cols[2][i];
            result[(i, 3)] = cols[3][i];
            result[(i, 4)] = cols[4][i];
            result[(i, 5)] = cols[5][i];
        }
        result
    }

    /// get the a submatrix from discard row `i` and column `j`
    ///
    fn get_submatrix(&self, selected: (usize, usize)) -> M55<T> {
        let mut values: [T; 25] = [T::zero(); 25];
        let mut result: M55<T> = M55::zeros();
        let mut index: usize = 0;
        for i in 0..6 {
            for j in 0..6 {
                if !(i == selected.0 || j == selected.1) {
                    values[index] = self[(i, j)];
                    index += 1;
                }
            }
        }
        index = 0;
        for r in 0.. 5 {
            for c in 0..5 {
                result[(r, c)] = values[index];
                index += 1;
            }
        }
        result
    }

    pub fn get_rows(self) -> V6<V6<T>> {
        let mut r0 = V6::zeros();
        let mut r1 = V6::zeros();
        let mut r2 = V6::zeros();
        let mut r3 = V6::zeros();
        let mut r4 = V6::zeros();
        let mut r5 = V6::zeros();

        for j in 0..self.rows() {
            r0[j] = self[(0, j)];
            r1[j] = self[(1, j)];
            r2[j] = self[(2, j)];
            r3[j] = self[(3, j)];
            r4[j] = self[(4, j)];
            r5[j] = self[(5, j)];
        }
        V6::new([r0, r1, r2, r3, r4, r5])
    }

    pub fn get_cols(self) -> V6<V6<T>> {
        let mut c0 = V6::zeros();
        let mut c1 = V6::zeros();
        let mut c2 = V6::zeros();
        let mut c3 = V6::zeros();
        let mut c4 = V6::zeros();
        let mut c5 = V6::zeros();

        for i in 0..self.cols() {
            c0[i] = self[(i, 0)];
            c1[i] = self[(i, 1)];
            c2[i] = self[(i, 2)];
            c3[i] = self[(i, 3)];
            c4[i] = self[(i, 4)];
            c5[i] = self[(i, 5)];
        }
        V6::new([c0, c1, c2, c3, c4, c5])
    }

    pub fn get_upper_triangular(&self) -> [T; 15] {
        let zero = T::zero();
        let mut result: [T; 15] = [zero; 15];
        let mut index = 0;
        for i in 0..self.rows() {
            for j in 0..self.cols() {
                if i < j {
                    result[index] = self[(i, j)];
                    index += 1;
                }
            }
        }
        result
    }

    pub fn get_lower_triangular(&self) -> [T; 15] {
        let zero = T::zero();
        let mut result: [T; 15] = [zero; 15];
        let mut index = 0;
        for i in 0..self.rows() {
            for j in 0..self.cols() {
                if i > j {
                    result[index] = self[(i, j)];
                    index += 1;
                }
            }
        }
        result
    }

    /// Applies `f` of each element in the M66
    pub fn for_each(&self, f: impl Fn(T) -> T) -> Self {
        let mut result = Self::zeros();
        for i in 0..self.rows() {
            for j in 0..self.cols() {
                result[(i, j)] = f(self[(i, j)]);
            }
        }
        result
    }

    /// Copy elements from a M33<T> matrix in one of the 4 cuadrants of the actual
    /// matrix, where the quadrants are enumerated as:
    /// +---+---+
    /// | 1 | 2 |
    /// +---+---+
    /// | 3 | 4 |
    /// +---+---+
    ///
    /// Function arguments:
    /// `r`: M33<T> submatrix elements to copy
    ///
    /// `quadrant`: number to choose the cuadrant where do the copy of the elements
    ///
    pub fn copy_elements_from(&mut self, r: &M33<T>, quadrant: usize) {
        match quadrant {
            1 => {
                for i in 0..3 {
                    for j in 0..3 {
                        self[(i, j)] = r[(i, j)];
                    }
                }
            },
            2 => {
                for i in 0..3 {
                    for j in 3..6 {
                        self[(i, j)] = r[(i, j-3)];
                    }
                }
            },
            3 => {
                for i in 3..6 {
                    for j in 0..3 {
                        self[(i, j)] = r[(i-3, j)];
                    }
                }
            },
            4 => {
                for i in 3..6 {
                    for j in 3..6 {
                        self[(i, j)] = r[(i-3, j-3)];
                    }
                }
            },
            _ => panic!("no more than 4 quadrants !!!")
        }
    }
}

impl<T: Num + Copy> Zero for M66<T> {
    fn zero() -> M66<T> {
        M66::new([[T::zero(); 6]; 6])
    }

    fn is_zero(&self) -> bool {
        *self == M66::zero()
    }
}

impl<T: Num + Copy> One for M66<T> {
    /// Create an identity matrix
    fn one() -> M66<T> {
        let one = T::one();
        let zero = T::zero();
        M66::new([
            [one, zero, zero, zero, zero, zero],
            [zero, one, zero, zero, zero, zero],
            [zero, zero, one, zero, zero, zero],
            [zero, zero, zero, one, zero, zero],
            [zero, zero, zero, zero, one, zero],
            [zero, zero, zero, zero, zero, one],
        ])
    }
}
//
impl<T> Deref for M66<T> {
    type Target = [[T; 6]; 6];
    #[inline]
    fn deref(&self) -> &Self::Target {
        &self.0
    }
}

impl<T> DerefMut for M66<T> {
    #[inline]
    fn deref_mut(&mut self) -> &mut Self::Target {
        &mut self.0
    }
}

impl<T> From<[[T; 6]; 6]> for M66<T> {
    fn from(data: [[T; 6]; 6]) -> M66<T> {
        M66(data)
    }
}

impl<T> Index<(usize, usize)> for M66<T> {
    type Output = T;
    #[inline(always)]
    fn index(&self, index: (usize, usize)) -> &T {
        &self.0[index.0][index.1]
    }
}

impl<T> IndexMut<(usize, usize)> for M66<T> {
    #[inline(always)]
    fn index_mut(&mut self, index: (usize, usize)) -> &mut T {
        &mut self.0[index.0][index.1]
    }
}

//-------------------------------------------------------------------------
//                        macros
//-------------------------------------------------------------------------
#[macro_export]
macro_rules! m66_new {
    ($($first_row:expr),*;
     $($second_row:expr),*;
     $($third_row:expr),*;
     $($fourth_row:expr),*;
     $($fifth_row:expr),*;
     $($sixth_row:expr),*
     ) => {
        M66::new([[$($first_row),*],
                 [$($second_row),*],
                 [$($third_row),*],
                 [$($fourth_row),*],
                 [$($fifth_row),*],
                 [$($sixth_row),*]])
    }
}

//-------------------------------------------------------------------------
//                        Display
//-------------------------------------------------------------------------
impl<T: Num + fmt::Display> fmt::Display for M66<T> {
    fn fmt(&self, dest: &mut fmt::Formatter) -> fmt::Result {
        println!();
        write!(
            dest,
            "|{0:<7.2} {1:^7.2} {2:^7.2} {3:^7.2} {4:^7.2} {5:>7.2}|\n",
            self[(0, 0)],
            self[(0, 1)],
            self[(0, 2)],
            self[(0, 3)],
            self[(0, 4)],
            self[(0, 5)]
        )?;
        write!(
            dest,
            "|{0:<7.2} {1:^7.2} {2:^7.2} {3:^7.2} {4:^7.2} {5:>7.2}|\n",
            self[(1, 0)],
            self[(1, 1)],
            self[(1, 2)],
            self[(1, 3)],
            self[(1, 4)],
            self[(1, 5)]
        )?;
        write!(
            dest,
            "|{0:<7.2} {1:^7.2} {2:^7.2} {3:^7.2} {4:^7.2} {5:>7.2}|\n",
            self[(2, 0)],
            self[(2, 1)],
            self[(2, 2)],
            self[(2, 3)],
            self[(2, 4)],
            self[(2, 5)]
        )?;
        write!(
            dest,
            "|{0:<7.2} {1:^7.2} {2:^7.2} {3:^7.2} {4:^7.2} {5:>7.2}|\n",
            self[(3, 0)],
            self[(3, 1)],
            self[(3, 2)],
            self[(3, 3)],
            self[(3, 4)],
            self[(3, 5)]
        )?;
        write!(
            dest,
            "|{0:<7.2} {1:^7.2} {2:^7.2} {3:^7.2} {4:^7.2} {5:>7.2}|\n",
            self[(4, 0)],
            self[(4, 1)],
            self[(4, 2)],
            self[(4, 3)],
            self[(4, 4)],
            self[(4, 5)]
        )?;
        write!(
            dest,
            "|{0:<7.2} {1:^7.2} {2:^7.2} {3:^7.2} {4:^7.2} {5:>7.2}|\n",
            self[(5, 0)],
            self[(5, 1)],
            self[(5, 2)],
            self[(5, 3)],
            self[(5, 4)],
            self[(5, 5)]
        )
    }
}


//-------------------------------------------------------------------------
//                        tests
//-------------------------------------------------------------------------
#[cfg(test)]
mod test_matrix6x6 {

    use crate::traits::LinearAlgebra;
    use crate::matrix6x6::M66;
    use crate::utils::nearly_equal;
    use crate::utils::compare_vecs;
    use crate::vector6::V6;

    // TODO(elsuizo:2020-06-02): ver porque tenemos que bajar el EPS para que ande inverse()
    const EPS: f32 = 1e-4;

    #[test]
    fn sub_test() {
        let m1 = m66_new!( 0.0,  1.0,  2.0,  3.0,  4.0,  5.0;
                           6.0,  7.0,  8.0,  9.0, 10.0, 11.0;
                          12.0, 13.0, 14.0, 15.0, 16.0, 17.0;
                          18.0, 19.0, 20.0, 21.0, 22.0, 23.0;
                          24.0, 25.0, 26.0, 27.0, 28.0, 29.0;
                          30.0, 31.0, 32.0, 33.0, 34.0, 35.0);

        let m2 = m66_new!( 0.0,  1.0,  2.0,  3.0,  4.0,  5.0;
                           6.0,  7.0,  8.0,  9.0, 10.0, 11.0;
                          12.0, 13.0, 14.0, 15.0, 16.0, 17.0;
                          18.0, 19.0, 20.0, 21.0, 22.0, 23.0;
                          24.0, 25.0, 26.0, 27.0, 28.0, 29.0;
                          30.0, 31.0, 32.0, 33.0, 34.0, 35.0);

        let result = m1 - m2;
        let expected = m66_new!( 0.0,  0.0,  0.0,  0.0,  0.0,  0.0;
                                 0.0,  0.0,  0.0,  0.0,  0.0,  0.0;
                                 0.0,  0.0,  0.0,  0.0,  0.0,  0.0;
                                 0.0,  0.0,  0.0,  0.0,  0.0,  0.0;
                                 0.0,  0.0,  0.0,  0.0,  0.0,  0.0;
                                 0.0,  0.0,  0.0,  0.0,  0.0,  0.0);

        assert!(compare_vecs(&result.as_vec(), &expected.as_vec(), EPS));
    }

    #[test]
    fn sum_test() {
        let m1 = m66_new!( 0.0,  1.0,  2.0,  3.0,  4.0,  5.0;
                           6.0,  7.0,  8.0,  9.0, 10.0, 11.0;
                          12.0, 13.0, 14.0, 15.0, 16.0, 17.0;
                          18.0, 19.0, 20.0, 21.0, 22.0, 23.0;
                          24.0, 25.0, 26.0, 27.0, 28.0, 29.0;
                          30.0, 31.0, 32.0, 33.0, 34.0, 35.0);

        let m2 = m66_new!( 0.0,  1.0,  2.0,  3.0,  4.0,  5.0;
                           6.0,  7.0,  8.0,  9.0, 10.0, 11.0;
                          12.0, 13.0, 14.0, 15.0, 16.0, 17.0;
                          18.0, 19.0, 20.0, 21.0, 22.0, 23.0;
                          24.0, 25.0, 26.0, 27.0, 28.0, 29.0;
                          30.0, 31.0, 32.0, 33.0, 34.0, 35.0);

        let result = m1 + m2;

        let expected = m66_new!( 0.0,  2.0,  4.0,   6.0,  8.0, 10.0;
                                12.0, 14.0, 16.0,  18.0, 20.0, 22.0;
                                24.0, 26.0, 28.0,  30.0, 32.0, 34.0;
                                36.0, 38.0, 40.0,  42.0, 44.0, 46.0;
                                48.0, 50.0, 52.0,  54.0, 56.0, 58.0;
                                60.0, 62.0, 64.0,  66.0, 68.0, 70.0);

        assert!(compare_vecs(&result.as_vec(), &expected.as_vec(), EPS));
    }

    #[test]
    fn matrix6x6_det_test() {

        use super::test_matrix6x6::EPS;

        // let m: M66<f64> = Matrix6x6::zeros();
        let m = M66::new([
            [1.0, 1.0, 3.0, 4.0, 9.0, 3.0],
            [10.0, 10.0, 1.0, 2.0, 2.0, 5.0],
            [2.0, 9.0, 6.0, 10.0, 10.0, 9.0],
            [10.0, 9.0, 9.0, 7.0, 3.0, 6.0],
            [7.0, 6.0, 6.0, 2.0, 9.0, 5.0],
            [3.0, 8.0, 1.0, 4.0, 1.0, 5.0],
        ]);
        let result = m.det();
        let expected = 3271.9999999999723;
        assert!(nearly_equal(result, expected, EPS));
    }

    #[test]
    fn matrix6x6_mul_test() {

        use super::test_matrix6x6::EPS;

        let m = M66::new([
            [0.0, 1.0, 2.0, 3.0, 4.0, 5.0],
            [6.0, 7.0, 8.0, 9.0, 10.0, 11.0],
            [12.0, 13.0, 14.0, 15.0, 16.0, 17.0],
            [18.0, 19.0, 20.0, 21.0, 22.0, 23.0],
            [24.0, 25.0, 26.0, 27.0, 28.0, 29.0],
            [30.0, 31.0, 32.0, 33.0, 34.0, 35.0],
        ]);
        let result = m * m;
        let expected = M66::new([
            [330.0, 345.0, 360.0, 375.0, 390.0, 405.0],
            [870.0, 921.0, 972.0, 1023.0, 1074.0, 1125.0],
            [1410.0, 1497.0, 1584.0, 1671.0, 1758.0, 1845.0],
            [1950.0, 2073.0, 2196.0, 2319.0, 2442.0, 2565.0],
            [2490.0, 2649.0, 2808.0, 2967.0, 3126.0, 3285.0],
            [3030.0, 3225.0, 3420.0, 3615.0, 3810.0, 4005.0],
        ]);

        assert!(compare_vecs(&result.as_vec(), &expected.as_vec(), EPS));
    }

    #[test]
    fn matrix6x6_norm2_test() {

        use super::test_matrix6x6::EPS;

        let m = m66_new!( 0.0,  1.0,  2.0,  3.0,  4.0,  5.0;
                          6.0,  7.0,  8.0,  9.0, 10.0, 11.0;
                         12.0, 13.0, 14.0, 15.0, 16.0, 17.0;
                         18.0, 19.0, 20.0, 21.0, 22.0, 23.0;
                         24.0, 25.0, 26.0, 27.0, 28.0, 29.0;
                         30.0, 31.0, 32.0, 33.0, 34.0, 35.0);

        let result = m.norm2();
        let expected = 122.10651088291729;
        assert!(nearly_equal(result, expected, EPS));
    }

    #[test]
    fn matrix6x6_inv_test() {

        use super::test_matrix6x6::EPS;

        let m = m66_new!( 1.0,  1.0, 3.0,  4.0,  9.0, 3.0;
                         10.0, 10.0, 1.0,  2.0,  2.0, 5.0;
                          2.0,  9.0, 6.0, 10.0, 10.0, 9.0;
                         10.0,  9.0, 9.0,  7.0,  3.0, 6.0;
                          7.0,  6.0, 6.0,  2.0,  9.0, 5.0;
                          3.0,  8.0, 1.0,  4.0,  1.0, 5.0);

        let expected = m66_new!(-0.538814,  0.577934,  0.665342, -0.0837408, -0.169621, -1.18215;
                                  2.16076,  -1.52751,  -2.44071,    0.44132,  0.324572,  3.77017;
                                 0.214548, -0.415037, -0.394254,   0.147922,  0.197433, 0.621027;
                                 0.700183, -0.240526, -0.525978,   0.203545, -0.211797, 0.734719;
                                  0.85055, -0.471577, -0.827934,   0.110636,  0.114609,  1.20416;
                                 -3.90709,   2.46699,   4.17115,  -0.870416, -0.310513, -6.07579);

        if let Some(result) = m.inverse() {
            assert!(compare_vecs(&result.as_vec(), &expected.as_vec(), EPS));
        }
    }

    #[test]
    fn inverse_fail() {
        let m = m66_new!( 1.0,  1.0, 3.0,  4.0,  9.0, 3.0;
                         10.0, 10.0, 1.0,  2.0,  2.0, 5.0;
                          2.0,  9.0, 6.0, 10.0, 10.0, 9.0;
                          2.0,  9.0, 6.0, 10.0, 10.0, 9.0;
                          7.0,  6.0, 6.0,  2.0,  9.0, 5.0;
                          3.0,  8.0, 1.0,  4.0,  1.0, 5.0);

        let result = m.inverse();
        let expected = None;
        assert_eq!(result, expected);

    }

    #[test]
    fn mul_vec_rhs() {
        let m = m66_new!( 1.0,  1.0, 3.0,  4.0,  9.0, 3.0;
                         10.0, 10.0, 1.0,  2.0,  2.0, 5.0;
                          2.0,  9.0, 6.0, 10.0, 10.0, 9.0;
                         10.0,  9.0, 9.0,  7.0,  3.0, 6.0;
                          7.0,  6.0, 6.0,  2.0,  9.0, 5.0;
                          3.0,  8.0, 1.0,  4.0,  1.0, 5.0);

        let v = V6::new([0.0, 1.0, 2.0, 3.0, 4.0, 5.0]);
        let result = m * v;

        let expected = V6::new([70.0, 51.0, 136.0, 90.0, 85.0, 51.0]);

        assert_eq!(
            &result[..],
            &expected[..],
            "\nExpected\n{:?}\nfound\n{:?}",
            &result[..],
            &expected[..]
        );


    }

    #[test]
    fn get_cols_test() {
        let m = m66_new!( 1.0,  1.0, 3.0,  4.0,  9.0, 3.0;
                         10.0, 10.0, 1.0,  2.0,  2.0, 5.0;
                          2.0,  9.0, 6.0, 10.0, 10.0, 9.0;
                         10.0,  9.0, 9.0,  7.0,  3.0, 6.0;
                          7.0,  6.0, 6.0,  2.0,  9.0, 5.0;
                          3.0,  8.0, 1.0,  4.0,  1.0, 5.0);

        let result = m.get_cols();
        let expected0 = V6::new([1.0, 10.0, 2.0, 10.0, 7.0, 3.0]);
        let expected1 = V6::new([1.0, 10.0, 9.0, 9.0, 6.0, 8.0]);
        let expected2 = V6::new([3.0, 1.0, 6.0, 9.0, 6.0, 1.0]);
        let expected3 = V6::new([4.0, 2.0, 10.0, 7.0, 2.0, 4.0]);
        let expected4 = V6::new([9.0, 2.0, 10.0, 3.0, 9.0, 1.0]);
        let expected5 = V6::new([3.0, 5.0, 9.0, 6.0, 5.0, 5.0]);

        let expected = V6::new([expected0, expected1, expected2, expected3, expected4, expected5]);
        assert_eq!(
            &result[..],
            &expected[..],
            "\nExpected\n{:?}\nfound\n{:?}",
            &result[..],
            &expected[..]
        );
    }

    #[test]
    fn get_rows_test() {
        let m = m66_new!( 1.0,  1.0, 3.0,  4.0,  9.0, 3.0;
                         10.0, 10.0, 1.0,  2.0,  2.0, 5.0;
                          2.0,  9.0, 6.0, 10.0, 10.0, 9.0;
                         10.0,  9.0, 9.0,  7.0,  3.0, 6.0;
                          7.0,  6.0, 6.0,  2.0,  9.0, 5.0;
                          3.0,  8.0, 1.0,  4.0,  1.0, 5.0);

        let result = m.transpose().get_rows();

        let expected0 = V6::new([1.0, 10.0, 2.0, 10.0, 7.0, 3.0]);
        let expected1 = V6::new([1.0, 10.0, 9.0, 9.0, 6.0, 8.0]);
        let expected2 = V6::new([3.0, 1.0, 6.0, 9.0, 6.0, 1.0]);
        let expected3 = V6::new([4.0, 2.0, 10.0, 7.0, 2.0, 4.0]);
        let expected4 = V6::new([9.0, 2.0, 10.0, 3.0, 9.0, 1.0]);
        let expected5 = V6::new([3.0, 5.0, 9.0, 6.0, 5.0, 5.0]);

        let expected = V6::new([expected0, expected1, expected2, expected3, expected4, expected5]);
        assert_eq!(
            &result[..],
            &expected[..],
            "\nExpected\n{:?}\nfound\n{:?}",
            &result[..],
            &expected[..]
        );
    }

    #[test]
    fn new_from_vecs_test() {
        let expected = m66_new!( 1.0,  1.0, 3.0,  4.0,  9.0, 3.0;
                                10.0, 10.0, 1.0,  2.0,  2.0, 5.0;
                                 2.0,  9.0, 6.0, 10.0, 10.0, 9.0;
                                10.0,  9.0, 9.0,  7.0,  3.0, 6.0;
                                 7.0,  6.0, 6.0,  2.0,  9.0, 5.0;
                                 3.0,  8.0, 1.0,  4.0,  1.0, 5.0);


        let cols = expected.get_cols();

        let result = M66::new_from_vecs(cols);

        assert!(compare_vecs(&result.as_vec(), &expected.as_vec(), EPS));
    }

    #[test]
    fn qr_test() {
        let expected = m66_new!( 1.0,  1.0, 3.0,  4.0,  9.0, 3.0;
                                10.0, 10.0, 1.0,  2.0,  2.0, 5.0;
                                 2.0,  9.0, 6.0, 10.0, 10.0, 9.0;
                                10.0,  9.0, 9.0,  7.0,  3.0, 6.0;
                                 7.0,  6.0, 6.0,  2.0,  9.0, 5.0;
                                 3.0,  8.0, 1.0,  4.0,  1.0, 5.0);
        if let Some((q, r)) = expected.qr() {
            let result = q * r;
            assert!(compare_vecs(&result.as_vec(), &expected.as_vec(), EPS));
            assert!(nearly_equal(q.det().abs(), 1.0, EPS));
        }
    }

    #[test]
    fn get_diagonal_test() {
        let m = m66_new!( 1.0,  1.0, 3.0,  4.0,  9.0, 3.0;
                         10.0, 10.0, 1.0,  2.0,  2.0, 5.0;
                          2.0,  9.0, 6.0, 10.0, 10.0, 9.0;
                         10.0,  9.0, 9.0,  7.0,  3.0, 6.0;
                          7.0,  6.0, 6.0,  2.0,  9.0, 5.0;
                          3.0,  8.0, 1.0,  4.0,  1.0, 5.0);
        let result = m.get_diagonal();
        let expected = V6::new([1.0, 10.0, 6.0, 7.0, 9.0, 5.0]);
        assert_eq!(
            &result[..],
            &expected[..],
            "\nExpected\n{:?}\nfound\n{:?}",
            &result[..],
            &expected[..]
        );
    }

    #[test]
    fn get_upper_triangular_test() {
        let m = m66_new!( 1.0,  1.0, 3.0,  4.0,  9.0, 3.0;
                         10.0, 10.0, 1.0,  2.0,  2.0, 5.0;
                          2.0,  9.0, 6.0, 10.0, 10.0, 9.0;
                         10.0,  9.0, 9.0,  7.0,  3.0, 6.0;
                          7.0,  6.0, 6.0,  2.0,  9.0, 5.0;
                          3.0,  8.0, 1.0,  4.0,  1.0, 5.0);
        let result = m.get_upper_triangular();
        let expected = [1.0, 3.0, 4.0, 9.0, 3.0, 1.0, 2.0, 2.0, 5.0, 10.0, 10.0, 9.0, 3.0, 6.0, 5.0];
        assert_eq!(
            &result[..],
            &expected[..],
            "\nExpected\n{:?}\nfound\n{:?}",
            &result[..],
            &expected[..]
        );
    }

    #[test]
    fn get_lower_triangular_test() {
        let m = m66_new!( 1.0,  1.0, 3.0,  4.0,  9.0, 3.0;
                         10.0, 10.0, 1.0,  2.0,  2.0, 5.0;
                          2.0,  9.0, 6.0, 10.0, 10.0, 9.0;
                         10.0,  9.0, 9.0,  7.0,  3.0, 6.0;
                          7.0,  6.0, 6.0,  2.0,  9.0, 5.0;
                          3.0,  8.0, 1.0,  4.0,  1.0, 5.0);
        let result = m.get_lower_triangular();
        let expected = [10.0, 2.0, 9.0, 10.0, 9.0, 9.0, 7.0, 6.0, 6.0, 2.0, 3.0, 8.0, 1.0, 4.0, 1.0];
        assert_eq!(
            &result[..],
            &expected[..],
            "\nExpected\n{:?}\nfound\n{:?}",
            &result[..],
            &expected[..]
        );
    }

    #[test]
    fn for_each_test() {
        let m = m66_new!( 1.0,  1.0, 3.0,  4.0,  9.0, 3.0;
                         10.0, 10.0, 1.0,  2.0,  2.0, 5.0;
                          2.0,  9.0, 6.0, 10.0, 10.0, 9.0;
                         10.0,  9.0, 9.0,  7.0,  3.0, 6.0;
                          7.0,  6.0, 6.0,  2.0,  9.0, 5.0;
                          3.0,  8.0, 1.0,  4.0,  1.0, 5.0);

        let result = m.for_each(|element| element + 1.0);

        let expected = m66_new!( 2.0,  2.0, 4.0,  5.0, 10.0, 4.0;
                                11.0, 11.0, 2.0,  3.0,  3.0, 6.0;
                                 3.0, 10.0, 7.0, 11.0, 11.0,10.0;
                                11.0, 10.0,10.0,  8.0,  4.0, 7.0;
                                 8.0,  7.0, 7.0,  3.0, 10.0, 6.0;
                                 4.0,  9.0, 2.0,  5.0,  2.0, 6.0);

        assert!(compare_vecs(&result.as_vec(), &expected.as_vec(), EPS));
    }
}
