// -*- coding: utf-8 -*-
// ------------------------------------------------------------------------------------------------
// Copyright © 2021, stack-graphs authors.
// Licensed under either of Apache License, Version 2.0, or MIT license, at your option.
// Please see the LICENSE-APACHE or LICENSE-MIT files in this distribution for license details.
// ------------------------------------------------------------------------------------------------

use libc::c_char;
use stack_graphs::arena::Handle;
use stack_graphs::c::sg_stack_graph_add_symbols;
use stack_graphs::c::sg_stack_graph_free;
use stack_graphs::c::sg_stack_graph_new;
use stack_graphs::c::sg_stack_graph_symbols;
use stack_graphs::c::sg_symbol_handle;
use stack_graphs::c::sg_symbols;
use stack_graphs::graph::Symbol;

fn strings(data: &[&'static str]) -> Vec<*const c_char> {
    data.iter()
        .map(|s| s.as_bytes().as_ptr() as *const c_char)
        .collect()
}

fn lengths(data: &[&'static str]) -> Vec<usize> {
    data.iter().map(|s| s.len()).collect()
}

fn get_symbol(arena: &sg_symbols, handle: Handle<Symbol>) -> &str {
    let slice = unsafe { std::slice::from_raw_parts(arena.symbols, arena.count) };
    let symbol = &slice[handle.as_usize()];
    unsafe {
        let bytes = std::slice::from_raw_parts(symbol.symbol as *const u8, symbol.symbol_len);
        std::str::from_utf8_unchecked(bytes)
    }
}

#[test]
fn can_create_symbols() {
    let graph = sg_stack_graph_new();

    let symbol_data = ["a", "a", "b", "c"];
    let mut handles: [Option<Handle<Symbol>>; 4] = [None; 4];
    sg_stack_graph_add_symbols(
        graph,
        symbol_data.len(),
        strings(&symbol_data).as_ptr(),
        lengths(&symbol_data).as_ptr(),
        handles.as_mut_ptr() as *mut sg_symbol_handle,
    );

    // All of the symbols should have been created successfully
    assert!(handles.as_ref().iter().all(|h| h.is_some()));

    // The handles should be comparable.
    let a1 = handles[0].unwrap();
    let a2 = handles[1].unwrap();
    let b = handles[2].unwrap();
    let c = handles[3].unwrap();
    assert_eq!(a1, a2);
    assert_ne!(a1, b);
    assert_ne!(a1, c);
    assert_ne!(a2, b);
    assert_ne!(a2, c);
    assert_ne!(b, c);

    // We should be able to dereference into the symbols arena to get the symbol content.
    let symbol_arena = sg_stack_graph_symbols(graph);
    assert_eq!(get_symbol(&symbol_arena, a1), "a");
    assert_eq!(get_symbol(&symbol_arena, a2), "a");
    assert_eq!(get_symbol(&symbol_arena, b), "b");
    assert_eq!(get_symbol(&symbol_arena, c), "c");

    sg_stack_graph_free(graph);
}
