

pub trait SqlQuote<OUT>
where OUT:std::fmt::Display
{
    fn sql_quote(&self)->OUT;
}

pub struct SqlExpr<T:std::fmt::Display>(pub T);

impl <T:std::fmt::Display>SqlQuote<String> for SqlExpr<T>
{
    fn sql_quote(&self)->String{format!("{}",&self.0)}
}


macro_rules! sql_quote_self {
    ($in_type:ty) => {
        impl SqlQuote<$in_type> for $in_type{
            fn sql_quote(&self)->$in_type{*self}
        }
    };
}
macro_rules! sql_quote_vec {
    ($in_type:ty) => {
        impl SqlQuote<String> for $in_type
        {
            fn sql_quote(&self)->String{
                self.into_iter().map(|e|{
                    format!("{}",e.sql_quote())
                }).collect::<Vec<String>>()
                .join(",")
            }
        }
        impl SqlQuote<String> for &$in_type
        {
            fn sql_quote(&self)->String{
                self.into_iter().map(|e|{
                    format!("{}",e.sql_quote())
                }).collect::<Vec<String>>()
                .join(",")
            }
        }
    };
}


sql_quote_self!(i8);
sql_quote_self!(i16);
sql_quote_self!(i32);
sql_quote_self!(i64);
sql_quote_self!(i128);
sql_quote_self!(u8);
sql_quote_self!(u16);
sql_quote_self!(u32);
sql_quote_self!(u64);
sql_quote_self!(u128);
sql_quote_self!(f32);
sql_quote_self!(f64);
sql_quote_self!(usize);
sql_quote_self!(isize);
impl SqlQuote<String> for char{
    fn sql_quote(&self)->String{
        if (*self)=='\'' {
            "'\\''".to_string()
        } else {
            format!("'{}'",self)
        }
        
    }
}
impl SqlQuote<u8> for bool{
    fn sql_quote(&self)->u8{
        (*self) as u8
    }
}
impl SqlQuote<String> for &str{
    fn sql_quote(&self)->String{
        format!("'{}'",self.replace("'", "\\'"))
    }
}
impl SqlQuote<String> for String{
    fn sql_quote(&self)->String{
        format!("'{}'",self.replace("'", "\\'"))
    }
}



macro_rules! sql_quote_array {
    ($in_type:ty) => {
        sql_quote_vec!(Vec<$in_type>);
        sql_quote_vec!([$in_type]);
    };
}

sql_quote_array!(i8);
sql_quote_array!(i16);
sql_quote_array!(i32);
sql_quote_array!(i64);
sql_quote_array!(i128);
sql_quote_array!(u8);
sql_quote_array!(u16);
sql_quote_array!(u32);
sql_quote_array!(u64);
sql_quote_array!(u128);
sql_quote_array!(f32);
sql_quote_array!(f64);
sql_quote_array!(usize);
sql_quote_array!(isize);
sql_quote_array!(bool);
sql_quote_array!(&str);
sql_quote_array!(String);


#[macro_export]
//转义SQL生成,对单引号转义
macro_rules! sql_format {
    ($fmt:expr) => {
        format!($fmt)
    };
    ($fmt:expr,$($argsname:tt=$argsval:expr),+$(,)?) => {
        format!($fmt,$($argsname=$argsval.sql_quote()) ,+)
    };
    ($fmt:expr,$($args:expr),+$(,)?) => {
        format!($fmt,$($args.sql_quote()),+)
    };
}
#[macro_export]
//数组类型的SQL生成,当无元素时返回空字符串,有元素时返回对应SQL
macro_rules! sql_array_str {
    ($fmt:expr,$val:expr) => {
        if $val.len()==0 {SqlExpr("".to_string())}
        else {SqlExpr(format!($fmt,$val.sql_quote()))}
    };
}


#[test]
fn test_sql_format_macro(){
    
    let str_val1="1'1'1";
    let str_val2="1'1'1";
    let char_val1='\'';
    let char_val2='"';
    let i_val1=1;
    let ivac=vec![1,2,4];
    let evac:Vec<String>=vec![];
    let str_vac=vec!["1","1'1","2'2'2'2'"];
    let str_arr=["1","1'1","2'2'2'2'"];
    let str_arr1=["1".to_string(),"1'1".to_string(),"2'2'2'2'".to_string()];
    let aa=||{1};
    fn bb()->i8{1}
    let sql=sql_format!(
        "
{str_val1}
{str_val2}
{char_val1}
{char_val2}
{i_val1}
({ivac})
({str_vac})
({str_arr})
({str_arr1})
({str_arr11})
({str_arr12})
{in1_where}
{in2_where}
{use_data}
",
        str_val1=str_val1,
        str_val2=str_val2,
        char_val1=char_val1,
        char_val2=char_val2,
        i_val1=i_val1,
        ivac=ivac,
        str_vac=str_vac,
        str_arr=str_arr,
        str_arr1=str_arr1,
        str_arr11=aa(),
        str_arr12=bb(),
        in1_where=sql_array_str!("ddd in ({})",ivac),
        in2_where=sql_array_str!("ddd in ({})",evac),
        use_data=SqlExpr(1)
    );
    assert!(sql.as_str()==r#"
'1\'1\'1'
'1\'1\'1'
'\''
'"'
1
(1,2,4)
('1','1\'1','2\'2\'2\'2\'')
('1','1\'1','2\'2\'2\'2\'')
('1','1\'1','2\'2\'2\'2\'')
(1)
(1)
ddd in (1,2,4)

1
"#);
}