//! SQLite parser
use log::error;

pub mod ast;
pub mod parse {
    #![allow(unused_braces)]
    #![allow(unused_comparisons)] // FIXME
    #![allow(clippy::collapsible_if)]
    #![allow(clippy::if_same_then_else)]
    #![allow(clippy::absurd_extreme_comparisons)] // FIXME
    #![allow(clippy::needless_return)]

    include!(concat!(env!("OUT_DIR"), "/parse.rs"));
}

use ast::{Cmd, ExplainKind, Name, Stmt};

/// Parser context
pub struct Context {
    explain: Option<ExplainKind>,
    stmt: Option<Stmt>,
    constraint_name: Option<Name>, // transient
    done: bool,
    error: Option<String>,
}

impl Context {
    pub fn new() -> Context {
        Context {
            explain: None,
            stmt: None,
            constraint_name: None,
            done: false,
            error: None,
        }
    }

    /// Consume parsed command
    pub fn cmd(&mut self) -> Option<Cmd> {
        if let Some(stmt) = self.stmt.take() {
            match self.explain.take() {
                Some(ExplainKind::Explain) => Some(Cmd::Explain(stmt)),
                Some(ExplainKind::QueryPlan) => Some(Cmd::ExplainQueryPlan(stmt)),
                None => Some(Cmd::Stmt(stmt)),
            }
        } else {
            None
        }
    }

    fn constraint_name(&mut self) -> Option<Name> {
        self.constraint_name.take()
    }
    fn no_constraint_name(&self) -> bool {
        self.constraint_name.is_none()
    }

    fn sqlite3_error_msg(&mut self, msg: &str) {
        error!("parser error: {}", msg);
    }

    /// This routine is called after a single SQL statement has been parsed.
    fn sqlite3_finish_coding(&mut self) {
        self.done = true;
    }

    /// Return `true` if parser completes either sucessfully or with an error.
    pub fn done(&self) -> bool {
        self.done || self.error.is_some()
    }

    pub fn is_ok(&self) -> bool {
        self.error.is_none()
    }

    /// Consume error generated by parser
    pub fn error(&mut self) -> Option<String> {
        self.error.take()
    }

    pub fn reset(&mut self) {
        self.explain = None;
        self.stmt = None;
        self.constraint_name = None;
        self.done = false;
        self.error = None;
    }
}
