//! A CLI for managing sql-audit. Running this binary expects to find a file called "audit.toml"
//! in the current directory. It will use the config in that file in order to:
//! 1. Create a new schema called `sql_audit`
//! 2. Create a table in that schema called `audit`
//! 3. Create a new function called `sql_audit_trigger`
//! 4. Create a trigger on all tables in the `public` schema not excluded which inserts a new record
//!     into `sql_audit.audit` any time an insert, update, or delete occurs containing copies of
//!     all the data.
//!
//! ## Locals
//! When performing transactions within that `public` schema, you can set some local values in a
//! transaction to track additional data.
//! 1. `SET LOCAL sql_audit.app_user = 'some_app_user';` is useful for recording an
//!     application-level user you want to track changes against, separate from the database user.
//! 2. `SET LOCAL sql_audit.request_id = 'some request';` can be used for tracing multiple row
//!     changes against a single operation (e.g. web request).
//!
//! ## audit.toml
//! Here are the configuration values you can provide in your `audit.toml`
//! ```toml
//! database = "postgres://user:password@host:port/database"  # the database connection string
//! exclude_tables = ["some_table"]  # A list of tables to exclude.
//! ```

#![forbid(unsafe_code)]
#![deny(missing_docs)]
#![deny(clippy::pedantic)]
#![deny(clippy::cargo)]

use color_eyre::eyre::Result;
use serde::Deserialize;

#[async_std::main]
async fn main() -> Result<()> {
    let contents = std::fs::read_to_string("audit.toml")?;

    let Config {
        database,
        exclude_tables,
    } = toml::from_str(&contents)?;
    let pool = sqlx::PgPool::connect(&database).await?;
    sql_audit::generate_audit(&pool, exclude_tables).await
}

#[derive(Deserialize)]
struct Config {
    database: String,
    #[serde(default)]
    exclude_tables: Vec<String>,
}
