# runtime-loop

The runtime-loop crate provides a common runtime loop that continuously attempts to find changes in your system, and act upon them accordingly.

## Examples

```rust
extern crate runtime_loop;

use runtime_loop::{RuntimeLoop, Pollable, Signal};
use std::thread;
use std::sync::{Arc, Mutex};
use std::sync::atomic::Ordering;
use std::time;

// Create memory for the runtime loop. Grab the cancel flag to exit the loop based on time.
let mut rloop = RuntimeLoop::new();
let cancel_flag = rloop.get_cancel_flag();

// When polled, increment the value.
// This should normally be your poll function. We increment x as a test.
let target = Arc::new(Mutex::new(Pollable::new(0, Box::new(|x| x+1) ) ) );

// When the polled value changes, print a message.
if let Ok(mut target) = target.lock() {
    target.add_trigger(Signal::new(
        Box::new(|x| println!("Increment: {:?}", x))
    ));
}

// Add the pollable to the runtime loop.
rloop.add_pollable(target.clone());

// This is an async runtime loop.  Though it is acceptable to call rloop.run() without spawning a new thread, spawning a new thread will enable external cancellation of the runtime loop.
let runtime_thread = thread::spawn(move || {
   if let Result::Err(s) = rloop.run() {
       panic!(s); }
});

// Run for N milliseconds before canceling and joining threads.
thread::sleep(time::Duration::from_millis(100)); //let the thread run for a bit
cancel_flag.store(true, Ordering::Relaxed); //Cancel runtime loop!
if let Result::Err(s) = runtime_thread.join() {
   panic!(s);
}
```

## License

Licensed under either of

 * Apache License, Version 2.0, ([LICENSE-APACHE](LICENSE-APACHE) or http://www.apache.org/licenses/LICENSE-2.0)
 * MIT license ([LICENSE-MIT](LICENSE-MIT) or http://opensource.org/licenses/MIT)

at your option.

### Contribution

Unless you explicitly state otherwise, any contribution intentionally
submitted for inclusion in the work by you, as defined in the Apache-2.0
license, shall be dual licensed as above, without any additional terms or
conditions.
