//! The reduce function, and the replace function for applications
use crate::parse::{GrammarItem, ParseNode};

/// Reduces a syntax tree
pub fn reduce(tree: ParseNode) -> (ParseNode, bool) {
    if let GrammarItem::Application = tree.entry {
        match tree.children[0].entry {
            GrammarItem::Variable(_) => {
                let (right, stop) = reduce(tree.children[1].clone());
                if stop {
                    (tree, true)
                } else {
                    (
                        ParseNode {
                            children: vec![tree.children[0].clone(), right],
                            entry: GrammarItem::Application,
                        },
                        false,
                    )
                }
            }
            GrammarItem::Application => {
                let (left, stop) = reduce(tree.children[0].clone());
                if stop {
                    let (right, stop) = reduce(tree.children[1].clone());
                    if stop {
                        (tree, true)
                    } else {
                        (
                            ParseNode {
                                children: vec![tree.children[0].clone(), right],
                                entry: GrammarItem::Application,
                            },
                            false,
                        )
                    }
                } else {
                    (
                        ParseNode {
                            children: vec![left, tree.children[1].clone()],
                            entry: GrammarItem::Application,
                        },
                        false,
                    )
                }
            }
            GrammarItem::Lambda => {
                if let GrammarItem::Variable(v) = tree.children[0].children[0].entry {
                    (
                        replace(tree.children[0].children[1].clone(), v, &tree.children[1]),
                        false,
                    )
                } else {
                    panic!("no")
                }
            }
        }
    } else {
        (tree, true)
    }
}

/// Replaces all instances of a variable in a syntax tree with another tree
pub fn replace<'a>(tree: ParseNode<'a>, var: &str, new: &ParseNode<'a>) -> ParseNode<'a> {
    match tree.entry {
        GrammarItem::Variable(v) => {
            if v == var {
                new.clone()
            } else {
                tree
            }
        }
        GrammarItem::Lambda => {
            if let GrammarItem::Variable(v) = tree.children[0].entry {
                if v == var {
                    tree
                } else {
                    ParseNode {
                        children: {
                            vec![
                                tree.children[0].clone(),
                                replace(tree.children[1].clone(), var, &new),
                            ]
                        },
                        entry: tree.entry,
                    }
                }
            } else {
                ParseNode {
                    children: {
                        vec![
                            tree.children[0].clone(),
                            replace(tree.children[1].clone(), var, &new),
                        ]
                    },
                    entry: tree.entry,
                }
            }
        }
        GrammarItem::Application => ParseNode {
            children: {
                vec![
                    replace(tree.children[0].clone(), var, &new),
                    replace(tree.children[1].clone(), var, &new),
                ]
            },
            entry: tree.entry,
        },
    }
}
