//! The parser function and syntax-tree structs

use crate::lex::Token;
use colored::*;
use std::fmt;

/// A type of item in a syntax tree
#[derive(Debug, Clone)]
pub enum GrammarItem<'a> {
    Variable(&'a str),
    Lambda,
    Application,
}

/// A parsed structure, with a GrammarItem and a Vec of children.
#[derive(Debug, Clone)]
pub struct ParseNode<'a> {
    pub children: Vec<ParseNode<'a>>,
    pub entry: GrammarItem<'a>,
}

impl fmt::Display for ParseNode<'_> {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        let color = f.sign_plus();
        match self.entry {
            GrammarItem::Variable(v) => {
                if color {
                    write!(f, "{}", v.blue())
                } else {
                    write!(f, "{}", v)
                }
            }
            GrammarItem::Lambda => {
                if color {
                    write!(
                        f,
                        "{}{:+}.{:+}",
                        "λ".purple(),
                        self.children[0],
                        self.children[1]
                    )
                } else {
                    write!(f, "λ{}.{}", self.children[0], self.children[1])
                }
            }
            GrammarItem::Application => {
                if color {
                    write!(
                        f,
                        "({:+} {:+})",
                        self.children[0],
                        self.children[1]
                    )
                } else {
                    write!(
                        f,
                        "({} {})",
                        self.children[0],
                        self.children[1]
                    )
                }
            }
        }
    }
}

/// An error that occurs while parsing
pub enum ParseError {
    UnbalancedParens,
    EmptyProgram,
    InvalidExpression,
}

impl fmt::Display for ParseError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            ParseError::UnbalancedParens => {
                write!(f, "Unbalanced parenthesis")
            }
            ParseError::EmptyProgram => {
                write!(f, "Empty program")
            }
            ParseError::InvalidExpression => {
                write!(f, "Invalid expression")
            }
        }
    }
}

/// Checkes the paren balance of a token vector
pub fn check_balance(input: &[Token]) -> bool {
    let mut count = 0;

    for token in input {
        count += match token {
            Token::LeftParen => 1,
            Token::RightParen => -1,
            _ => 0,
        }
    }

    count == 0
}

/// Parses a token vector into a syntax tree
pub fn parse(input: &[Token]) -> Result<ParseNode, ParseError> {
    // Empty program
    if input.is_empty() {
        Err(ParseError::EmptyProgram)
    // Unbalanced parenthesis
    } else if !check_balance(input) {
        Err(ParseError::UnbalancedParens)
    // Application
    } else if input[0] == Token::LeftParen {
        // Complete application
        if input[input.len() - 1] == Token::RightParen {
            // ()
            if input.len() == 2 {
                Err(ParseError::InvalidExpression)
            // Left-hand var
            } else if let Token::Var(v) = &input[1] {
                // (x)
                if input.len() == 3 {
                    Ok(ParseNode {
                        children: vec![],
                        entry: GrammarItem::Variable(v),
                    })
                // (x tree)
                } else {
                    let argument = parse(&input[2..input.len() - 1])?;
                    Ok(ParseNode {
                        children: vec![
                            ParseNode {
                                children: vec![],
                                entry: GrammarItem::Variable(v),
                            },
                            argument,
                        ],
                        entry: GrammarItem::Application,
                    })
                }
            // Left-hand application
            } else if input[1] == Token::LeftParen {
                let mut end_pos = 3;
                while end_pos < input.len() {
                    if let Ok(left_arm) = parse(&input[1..end_pos]) {
                        if let Ok(right_arm) = parse(&input[end_pos..input.len() - 1]) {
                            return Ok(ParseNode {
                                children: vec![left_arm, right_arm],
                                entry: GrammarItem::Application,
                            });
                        } else {
                            return Err(ParseError::InvalidExpression);
                        }
                    }
                    end_pos += 1;
                }
                Err(ParseError::InvalidExpression)
            // Left-hand lambda
            } else if input[1] == Token::Lambda {
                let mut end_pos = 5;
                while end_pos < input.len() {
                    if let Ok(left_arm) = parse(&input[1..end_pos]) {
                        if let Ok(right_arm) = parse(&input[end_pos..input.len() - 1]) {
                            return Ok(ParseNode {
                                children: vec![left_arm, right_arm],
                                entry: GrammarItem::Application,
                            });
                        } else {
                            return Err(ParseError::InvalidExpression);
                        }
                    }
                    end_pos += 1;
                }
                Err(ParseError::InvalidExpression)
            } else {
                Err(ParseError::InvalidExpression)
            }
        } else {
            Err(ParseError::InvalidExpression)
        }
    // Lambda
    } else if input[0] == Token::Lambda {
        // Not garbage
        if input.len() >= 4 && input[2] == Token::Dot {
            if let Token::Var(v) = &input[1] {
                Ok(ParseNode {
                    children: vec![
                        ParseNode {
                            children: vec![],
                            entry: GrammarItem::Variable(v),
                        },
                        parse(&input[3..])?,
                    ],
                    entry: GrammarItem::Lambda,
                })
            } else {
                Err(ParseError::InvalidExpression)
            }
        } else {
            Err(ParseError::InvalidExpression)
        }
    // One token
    } else if input.len() == 1 {
        // x
        if let Token::Var(v) = &input[0] {
            Ok(ParseNode {
                children: vec![],
                entry: GrammarItem::Variable(v),
            })
        } else {
            Err(ParseError::InvalidExpression)
        }
    } else {
        Err(ParseError::InvalidExpression)
    }
}
