mod lex;
mod parse;
mod reduce;

use colored::*;
use lex::lex;
use parse::parse;
use reduce::reduce;
use std::env;
use std::io::Write;

// Version data
const VERSION: &'static str = env!("CARGO_PKG_VERSION");
const NAME: &'static str = env!("CARGO_PKG_NAME");
const DESCRIPTION: &'static str = env!("CARGO_PKG_DESCRIPTION");

fn main() {
    let argv: Vec<String> = env::args().collect();
    let debug = argv.contains(&String::from("--debug")) || argv.contains(&String::from("-d"));
    let step = argv.contains(&String::from("--step")) || argv.contains(&String::from("-s"));

    // --help --version
    if argv.contains(&String::from("--help"))
        || argv.contains(&String::from("-h"))
        || argv.contains(&String::from("--version"))
        || argv.contains(&String::from("-v"))
    {
        println!("{} version {}", NAME.green(), VERSION.blue());
        println!("{}", DESCRIPTION);
        println!("");
        println!("  --debug -d              Show lexing and parsing data");
        println!("  --step -s               Show individual steps in reduction");
        println!("  --help --version -h -v  Show this message");
        println!("");
    } else {
        loop {
            // Get input program
            let mut input = String::new();
            print!("{}> ", "rs-lambda".purple());
            std::io::stdout().flush().expect("Pain and suffering");
            std::io::stdin()
                .read_line(&mut input)
                .expect("Pain and suffering; again");
            // Break on Ctrl-D and "exit"
            if input.is_empty() {
                println!();
                break;
            };
            if input == "exit\n" {
                break;
            };
            // Lex, parse, and run the program
            match lex(input.trim()) {
                Ok(l) => {
                    if debug {
                        println!("{}", format!("Lexed: {:?}", l).blue());
                    }
                    match parse(&l) {
                        Ok(mut t) => {
                            if debug {
                                println!("{}", format!("Parsed: {:?}", t).blue());
                            }
                            if step {
                                println!("{} {:+}", "0:".blue(), t);
                            }
                            let mut i = 1;
                            loop {
                                let (tr, stop) = reduce(t);
                                t = tr;
                                if stop {
                                    break;
                                }
                                if step {
                                    println!("{}{} {:+}", i.to_string().blue(), ":".blue(), t);
                                }
                                i += 1;
                            }
                            println!("{:+}", t);
                        }
                        Err(e) => {
                            eprintln!("{}", format!("Error while parsing! {}.", e).red());
                        }
                    }
                }
                Err(e) => {
                    eprintln!("{}", format!("Error while lexing! {}.", e).red())
                }
            };
        }
    }
}
