//! The lexer function and Token structs

use std::fmt;

/// A token for use as output of lex
#[derive(Debug, PartialEq)]
pub enum Token {
    LeftParen,
    RightParen,
    Lambda,
    Dot,
    Var(String),
}

/// An error that occurs while lexing
pub enum LexError {
    UnknownToken(char),
}

impl fmt::Display for LexError {
    fn fmt(&self, f: &mut fmt::Formatter<'_>) -> fmt::Result {
        match self {
            LexError::UnknownToken(ch) => write!(f, "Unknown character '{}'", ch),
        }
    }
}

/// Turns a program string into a vector of tokens
pub fn lex(input: &str) -> Result<Vec<Token>, LexError> {
    let mut result: Vec<Token> = Vec::new();
    let mut it = input.chars().peekable();
    while let Some(&ch) = it.peek() {
        match ch {
            'a'..='z' | 'A'..='Z' => {
                let mut variable = String::new();
                while it.peek() != None && it.peek().unwrap().is_ascii_alphabetic() {
                    variable.push(*it.peek().unwrap());
                    it.next();
                }
                result.push(Token::Var(variable.to_lowercase()))
            }
            '(' | '[' | '{' => {
                result.push(Token::LeftParen);
                it.next();
            }
            ')' | ']' | '}' => {
                result.push(Token::RightParen);
                it.next();
            }
            '\\' | 'λ' => {
                result.push(Token::Lambda);
                it.next();
            }
            '.' | ':' => {
                result.push(Token::Dot);
                it.next();
            }
            ' ' => {
                it.next();
            }
            _ => return Err(LexError::UnknownToken(ch)),
        }
    }
    Ok(result)
}
