/* The MIT License (MIT)
 *
 * Copyright (c) 2015 William Orr <will@worrbase.com>
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in all
 * copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
#![no_std]
#![feature(convert, core_slice_ext, core_str_ext, no_std, unique)]

use core::mem;
use core::ptr::Unique;
use core::slice;

#[macro_export]
macro_rules! cstr {
    ($arg:expr) => ($crate::CString::new(concat!($arg, "\0")))
}

pub struct CString {
    data: Unique<[i8]>,
    len: usize
}

fn strlen(ptr: *const i8) -> isize {
    let mut ctr = 0isize;
    loop {
        if unsafe { *ptr.offset(ctr) == 0 as i8 } {
            break
        }

        ctr += 1;
    }

    ctr
}

impl CString {
    pub fn new(s: &str) -> CString {
        CString {
            data: unsafe { Unique::new(mem::transmute(s)) },
            len: s.len()
        }
    }

    pub unsafe fn from_raw(ptr: *mut i8) -> CString {
        let siz = strlen(ptr) as usize;
        let slic = slice::from_raw_parts(ptr, siz + 1);

        CString {
            data: Unique::new(mem::transmute(slic)),
            len: strlen(ptr) as usize
        }
    }

    pub unsafe fn into_raw(&mut self) -> *mut i8 {
        self.data.get_mut().as_mut_ptr()
    }
}

#[cfg(test)] #[macro_use] extern crate std;

#[cfg(test)]
mod test {
    use super::CString;
    use std::ffi::CString as OldString;

    #[test]
    fn new_cstring() {
        {
            let cstr = cstr!("foo");
            let buff = vec!['f' as i8,
                            'o' as i8,
                            'o' as i8,
                            0 as i8];

            unsafe { assert_eq!(cstr.data.get(), buff.as_slice()) };
        }
        {
            let mut raw = OldString::new("foo").unwrap().into_raw();
            let cstr = unsafe { CString::from_raw(raw) };
            let buff = vec!['f' as i8,
                            'o' as i8,
                            'o' as i8,
                            0 as i8];

            unsafe { assert_eq!(cstr.data.get(), buff.as_slice()) };
        }
    }

    #[test]
    fn from_cstring() {
        {
            let mut cstr = cstr!("foo");
            let raw = unsafe { cstr.into_raw() };
            let mut other = OldString::new("foo").unwrap().into_raw();

            unsafe {
                assert_eq!(*raw, *other);
            }
        }
    }
}
