#![no_std]
#![feature(convert, core_slice_ext, core_str_ext, libc, no_std, unique)]

extern crate libc;

use core::mem;
use core::ptr::Unique;
use core::slice;
use libc::c_char;

macro_rules! cstr {
    ($arg:expr) => (CString::new(concat!($arg, "\0")))
}

pub struct CString {
    data: Unique<[c_char]>,
    len: usize
}

fn strlen(ptr: *const c_char) -> isize {
    let mut ctr = 0isize;
    loop {
        if unsafe { *ptr.offset(ctr) == 0 as c_char } {
            break
        }

        ctr += 1;
    }

    ctr
}

impl CString {
    pub fn new(s: &str) -> CString {
        CString {
            data: unsafe { Unique::new(mem::transmute(s)) },
            len: s.len()
        }
    }

    pub unsafe fn from_raw(ptr: *mut c_char) -> CString {
        let siz = strlen(ptr) as usize;
        let slic = slice::from_raw_parts(ptr, siz + 1);

        CString {
            data: Unique::new(mem::transmute(slic)),
            len: strlen(ptr) as usize
        }
    }

    pub unsafe fn into_raw(&mut self) -> *mut c_char {
        self.data.get_mut().as_mut_ptr()
    }
}

#[cfg(test)] #[macro_use] extern crate std;

#[cfg(test)]
mod test {
    use super::CString;
    use std::ffi::CString as OldString;
    use libc::c_char;

    #[test]
    fn new_cstring() {
        {
            let cstr = cstr!("foo");
            let buff = vec!['f' as c_char,
                            'o' as c_char,
                            'o' as c_char,
                            0 as c_char];

            unsafe { assert_eq!(cstr.data.get(), buff.as_slice()) };
        }
        {
            let mut raw = OldString::new("foo").unwrap().into_raw();
            let cstr = unsafe { CString::from_raw(raw) };
            let buff = vec!['f' as c_char,
                            'o' as c_char,
                            'o' as c_char,
                            0 as c_char];

            unsafe { assert_eq!(cstr.data.get(), buff.as_slice()) };
        }
    }

    #[test]
    fn from_cstring() {
        {
            let mut cstr = cstr!("foo");
            let raw = unsafe { cstr.into_raw() };
            let mut other = OldString::new("foo").unwrap().into_raw();

            unsafe {
                assert_eq!(*raw, *other);
            }
        }
    }
}
