/*
 *  Quick Test: CLI for stress testing in competitive programming
 *  Copyright (C) 2021  Luis Miguel Báez
 *  License: MIT (See the LICENSE file in the repository root directory)
 */

use std::path::PathBuf;
use structopt::StructOpt;
use structopt::clap::ArgGroup;

/// CLI for stress testing in competitive programming contest
#[derive(StructOpt, Debug)]
#[structopt(name = "quicktest")]
pub enum Opt {

    /// Check TLE
    TLE {
        /// Target File
        #[structopt(short = "t", long = "target-file", parse(from_os_str))]
        target_file: PathBuf,

        /// Generator File
        #[structopt(short = "g", long = "gen-file", parse(from_os_str))]
        gen_file: PathBuf,

        /// Timeout TLE
        #[structopt(short = "o", long = "timeout", default_value = "2000")]
        timeout: u32,

        /// Number of test cases
        #[structopt(short = "n", long = "test-cases", default_value = "1000")]
        test_cases: u32,

        /// Break if Time Limit Exceeded (TLE) occurs
        #[structopt(short = "b", long = "tle-break")]
        tle_break: bool,

        /// Save only bad cases with WA, TLE or RTE states
        #[structopt(short = "s", long = "save-bad", conflicts_with="save-all")]
        save_bad: bool,

        /// Save all test cases
        #[structopt(short = "a", long = "save-all", conflicts_with="save-bad")]
        save_all: bool,

        /// Run all test cases
        #[structopt(long = "run-all")]
        run_all: bool,

        /// Run test cases Accepted
        #[structopt(long = "run-ac")]
        run_ac: bool,

        /// Run test cases Wrong Answer
        #[structopt(long = "run-wa")]
        run_wa: bool,

        /// Run test cases Time Limited Exceeded
        #[structopt(long = "run-tle")]
        run_tle: bool,

        /// Run test cases Run Time Error
        #[structopt(long = "run-rte")]
        run_rte: bool,
    },
    /// Check the correctness of the <target-file> comparing it with <slow-file> with input test generated by <gen-file>
    Cmp {
        /// Target File
        #[structopt(short = "t", long = "target-file", parse(from_os_str))]
        target_file: PathBuf,

        /// Correct File
        #[structopt(short = "c", long = "correct-file", parse(from_os_str))]
        correct_file: PathBuf,

        /// Generator File
        #[structopt(short = "g", long = "gen-file", parse(from_os_str))]
        gen_file: PathBuf,

        /// Timeout TLE
        #[structopt(short = "o", long = "timeout", default_value = "2000")]
        timeout: u32,

        /// Number of test cases
        #[structopt(short = "n", long = "test-cases", default_value = "1000")]
        test_cases: u32,

        /// Break if WA, TLE or RTE states occurs
        #[structopt(short = "b", long = "break_bad")]
        break_bad: bool,

        /// Save only bad cases with WA, TLE or RTE states
        #[structopt(short = "s", long = "save-bad", conflicts_with="save-all")]
        save_bad: bool,

        /// Save all test cases
        #[structopt(short = "a", long = "save-all", conflicts_with="save-bad")]
        save_all: bool,

        /// run all test cases
        #[structopt(long = "run-all")]
        run_all: bool,

        /// run test cases Accepted
        #[structopt(long = "run-ac")]
        run_ac: bool,

        /// run test cases Wrong Answer
        #[structopt(long = "run-wa")]
        run_wa: bool,

        /// run test cases Time Limited Exceeded
        #[structopt(long = "run-tle")]
        run_tle: bool,

        /// run test cases Run Time Error
        #[structopt(long = "run-rte")]
        run_rte: bool,
    },
    /// Check the correctness of the answer using a script <checker-file> (Special for cases that do not have a unique answer)
    Check {
        /// Target File
        #[structopt(short = "t", long = "target-file", parse(from_os_str))]
        target_file: PathBuf,

        /// Checker File
        #[structopt(short = "c", long = "checker-file", parse(from_os_str))]
        checker_file: PathBuf,

        /// Generator File
        #[structopt(short = "g", long = "gen-file", parse(from_os_str))]
        gen_file: PathBuf,

        /// Timeout TLE
        #[structopt(short = "o", long = "timeout", default_value = "2000")]
        timeout: u32,

        /// Number of test cases
        #[structopt(short = "n", long = "test-cases", default_value = "1000")]
        test_cases: u32,

        /// Break if WA, TLE or RTE states occurs
        #[structopt(short = "b", long = "break_bad")]
        break_bad: bool,

        /// Save only bad cases with WA, TLE or RTE states
        #[structopt(short = "s", long = "save-bad", conflicts_with="save-all")]
        save_bad: bool,

        /// Save all test cases
        #[structopt(short = "a", long = "save-all", conflicts_with="save-bad")]
        save_all: bool,

        /// run all test cases
        #[structopt(long = "run-all")]
        run_all: bool,

        /// run test cases Accepted
        #[structopt(long = "run-ac")]
        run_ac: bool,

        /// run test cases Wrong Answer
        #[structopt(long = "run-wa")]
        run_wa: bool,

        /// run test cases Time Limited Exceeded
        #[structopt(long = "run-tle")]
        run_tle: bool,

        /// run test cases Run Time Error
        #[structopt(long = "run-rte")]
        run_rte: bool,
    },
    /// Shows examples of the selected subcommand
    #[structopt(group = ArgGroup::with_name("cmd").required(true))]
    Example {
        /// Show examples of the tle subcommand
        #[structopt(long = "tle", group = "cmd")]
        tle: bool,

        /// Show examples of the cmp subcommand
        #[structopt(long = "cmp", group = "cmd")]
        cmp: bool,

        /// Show examples of the check subcommand
        #[structopt(long = "check", group = "cmd")]
        check: bool,
    }
}