/*
 *  Quick Test: CLI for stress testing in competitive programming
 *  Copyright (C) 2021  Luis Miguel Báez
 *  License: MIT (See the LICENSE file in the repository root directory)
 */

use std::path::PathBuf;
use structopt::StructOpt;

/// CLI for stress testing in competitive programming contest
#[derive(StructOpt, Debug)]
#[structopt(name = "quicktest")]
pub enum Opt {

    /// Check TLE
    TLE {
        /// Target File
        #[structopt(short = "t", long = "target-file", parse(from_os_str))]
        target_file: PathBuf,

        /// Generator File
        #[structopt(short = "g", long = "gen-file", parse(from_os_str))]
        gen_file: PathBuf,

        /// Timeout TLE
        #[structopt(short = "o", long = "timeout", default_value = "2000")]
        timeout: u32,

        /// Number of test cases
        #[structopt(short = "n", long = "test-cases", default_value = "1000")]
        test_cases: u32,

        /// Break if Time Limit Exceeded (TLE) occurs
        #[structopt(short = "b", long = "tle-break")]
        tle_break: bool,

        /// Save test cases
        #[structopt(short = "s", long = "save-cases")]
        save_cases: bool,
    },
    /// Check the correctness of the <target-file> comparing it with <slow-file> with input test generated by <gen-file>
    Cmp {
        /// Target File
        #[structopt(short = "t", long = "target-file", parse(from_os_str))]
        target_file: PathBuf,

        /// Correct File
        #[structopt(short = "c", long = "correct-file", parse(from_os_str))]
        correct_file: PathBuf,

        /// Generator File
        #[structopt(short = "g", long = "gen-file", parse(from_os_str))]
        gen_file: PathBuf,

        /// Timeout TLE
        #[structopt(short = "o", long = "timeout", default_value = "2000")]
        timeout: u32,

        /// Number of test cases
        #[structopt(short = "n", long = "test-cases", default_value = "1000")]
        test_cases: u32,

        /// Break if Wrong Answer (WA) occurs
        #[structopt(short = "b", long = "wa-break")]
        wa_break: bool,

        /// Save test cases
        #[structopt(short = "s", long = "save-cases")]
        save_cases: bool,
    },
    /// Check the correctness of the answer using a script <checker-file> (Special for cases that do not have a unique answer)
    Check {
        /// Target File
        #[structopt(short = "t", long = "target-file", parse(from_os_str))]
        target_file: PathBuf,

        /// Checker File
        #[structopt(short = "c", long = "checker-file", parse(from_os_str))]
        checker_file: PathBuf,

        /// Generator File
        #[structopt(short = "g", long = "gen-file", parse(from_os_str))]
        gen_file: PathBuf,

        /// Timeout TLE
        #[structopt(short = "o", long = "timeout", default_value = "2000")]
        timeout: u32,

        /// Number of test cases
        #[structopt(short = "n", long = "test-cases", default_value = "1000")]
        test_cases: u32,

        /// Break if Wrong Answer (WA) occurs
        #[structopt(short = "b", long = "wa-break")]
        wa_break: bool,

        /// Save test cases
        #[structopt(short = "s", long = "save-cases")]
        save_cases: bool,
    },
    /// Run the test cases that are locally
    Run {

        /// Target File
        #[structopt(short = "t", long = "target-file", parse(from_os_str))]
        target_file: PathBuf,

        /// Timeout TLE
        #[structopt(short = "o", long = "timeout", default_value = "2000")]
        timeout: u32,

        /// run all test cases
        #[structopt(short = "a", long = "all")]
        all: bool,

        /// run test cases Wrong Answer
        #[structopt(short = "w", long = "wa")]
        wa: bool,

        /// run test cases Time Limited Exceeded
        #[structopt(short = "l", long = "tle")]
        tle: bool,

        /// run test cases Run Time Error
        #[structopt(short = "r", long = "rte")]
        rte: bool
    },
}