use std::path::PathBuf;
use structopt::StructOpt;

/// CLI for stress testing in competitive programming contest
#[derive(StructOpt, Debug)]
#[structopt(name = "quicktest")]
pub enum Opt {

    /// Check TLE
    #[structopt(help = r#"
quicktest tle
Check that <target-file> does not have the TLE status using random test cases generated by <gen-file>

USAGE:
    $ quicktest tle --target-file "code/main.cpp" --gen-file "code/gen.cpp" --timeout=1000 --test-cases=100
FLAGS:
    -h, --help          Prints help information
    -s, --save-cases    Save test cases
    -b, --tle-break     TLE Break
    -V, --version       Prints version information

OPTIONS:
    -g, --gen-file <gen-file>          Generator File
    -t, --target-file <target-file>    Target File
    -n, --test-cases <test-cases>      Number of test cases [default: 10000]
    -o, --timeout <timeout>            Timeout TLE [default: 2000]"#)]
    TLE {
        /// Target File
        #[structopt(short = "t", long = "target-file", parse(from_os_str))]
        target_file: PathBuf,

        /// Generator File
        #[structopt(short = "g", long = "gen-file", parse(from_os_str))]
        gen_file: PathBuf,

        /// Timeout TLE
        #[structopt(short = "o", long = "timeout", default_value = "2000")]
        timeout: u32,

        /// Number of test cases
        #[structopt(short = "n", long = "test-cases", default_value = "10000")]
        test_cases: u32,

        /// TLE Break
        #[structopt(short = "b", long = "tle-break")]
        tle_break: bool,

        /// Save test cases
        #[structopt(short = "s", long = "save-cases")]
        save_cases: bool,
    },
    /// Compare correctness with a slower program
    #[structopt(help = "Check the correctness of the <target-file> comparing it with <slow-file> with input test generated by <gen-file>")]
    Compare {
        /// Target File
        #[structopt(short = "t", long = "target-file", parse(from_os_str))]
        target_file: PathBuf,

        /// Slow File
        #[structopt(short = "s", long = "slow-file", parse(from_os_str))]
        slow_file: PathBuf,

        /// Generator File
        #[structopt(short = "g", long = "gen-file", parse(from_os_str))]
        gen_file: PathBuf,

        /// Timeout TLE
        #[structopt(short = "o", long = "timeout", default_value = "2000")]
        timeout: u32,

        /// Number of test cases
        #[structopt(short = "n", long = "test-cases", default_value = "10000")]
        test_cases: u32,
    },
}