//! An ASCII byte string, encoded using the 
//! [Bencoding](http://bittorrent.org/beps/bep_0003.html#bencoding) format.
//! 
//! This module contains the [`BencodeString`] type, and related error types.
//!
//! [`BencodeString`]: struct.BencodeString.html

use bencode::Bencodable;
use bencode::BencodeError;
/// An ASCII byte string, encoded using the 
/// [Bencoding](http://bittorrent.org/beps/bep_0003.html#bencoding) format.
pub struct BencodeString {
    /// The length of the bencoded string.
    pub length: i64,
    /// The contents of the bencoded string.
    pub contents: Vec<u8>
}

impl Bencodable<String> for BencodeString
{
    /// Converts a BencodeString object into a bencoded string,
    /// in the `<length>:<byte string>` format.
    ///
    /// # Examples
    ///
    /// ```
    /// use oxidation_bencode::bencode::Bencodable;
    /// use oxidation_bencode::string::BencodeString;
    ///
    /// let origString = "12:Hello World!";
    /// let bs = BencodeString::from_bencode_string(origString).unwrap();
    /// 
    /// assert_eq!(bs.to_bencode_string(), "12:Hello World!");
    /// ```
    fn to_bencode_string(&self) -> String {
        let final_length = self.length.to_string();
        let final_contents = self.extract_content();
        let mut final_string = String::new();
        final_string.push_str(&final_length);
        final_string.push(':');
        final_string.push_str(&final_contents);

        final_string
    }

    /// Converts a bencode string literal into a [`BencodeString`]
    /// object.
    ///
    /// # Examples
    ///
    /// ```
    /// use oxidation_bencode::bencode::Bencodable;
    /// use oxidation_bencode::string::BencodeString;
    ///
    /// // Valid String
    /// let origString = "12:Hello World!";
    /// let bs = BencodeString::from_bencode_string(origString).unwrap();
    /// 
    /// assert_eq!(bs.extract_content(), "Hello World!");
    ///
    /// // Invalid String
    /// let origString = "Hello World!";
    /// let bs = BencodeString::from_bencode_string(origString);
    /// 
    /// assert!(bs.is_err());
    /// ```    
    ///
    /// [`BencodeString`]: struct.BencodeString.html
    fn from_bencode_string(str: &'static str) -> Result<BencodeString,BencodeError> {
        let mut split_string = (&str).split(':');
        let pos_count = split_string.next();
        let pos_contents = split_string.next();

        if pos_count.is_none() || pos_contents.is_none() || split_string.count() != 0 {
           Err(BencodeError::ParseError("string was encoded incorrectly and could not be converted"))
        } else {

            let actual_count = i64::from_str_radix(pos_count.unwrap(), 10);

            if actual_count.is_err() {
                Err(BencodeError::ParseError("count was unable to be parsed correctly"))
            } else {

                let final_count: i64 = actual_count.unwrap();
                let mut final_contents: Vec<u8> = Vec::new();

                for c in pos_contents.unwrap().chars() {
                    final_contents.push(c as u8);
                }

                Ok(BencodeString {
                    length: final_count,
                    contents: final_contents.clone()
                })
            }
        }
    }


    /// Extracts the contents of a [`BencodeString`] and returns them
    /// as an [`std::string::String`].
    ///
    /// #Examples
    /// 
    /// ```
    /// use oxidation_bencode::bencode::Bencodable;
    /// use oxidation_bencode::string::BencodeString;
    ///
    /// let bcoded_str = BencodeString::from_bencode_string("12:Hello World!").unwrap();
    ///
    /// let str = bcoded_str.extract_content();
    ///
    /// assert_eq!(&str, "Hello World!");
    /// ```
    ///
    /// [`std::string::String`]: https://doc.rust-lang.org/nightly/collections/string/struct.String.html
    /// [`BencodeString`]: struct.BencodeString.html
    fn extract_content(&self) -> String
    {
        let result = String::from_utf8(self.contents.clone());

        if result.is_err() {
            panic!("oxidation_bencode error: bytes in string are not ASCII or are incorrect");
        } else {
            result.unwrap()
        }
    }
}