pub struct BencodeString {
    /// The length of the bencoded string.
    pub length: i64,
    /// The contents of the bencoded string.
    pub contents: Vec<u8>
}

impl BencodeString {
    /// Converts a bencoded string into an std::string::String.
    ///
    /// #Examples
    /// 
    /// ```
    /// use oxidation_bencode::string::BencodeString;
    ///
    /// let bcoded_str = BencodeString::from_string("5:hello");
    ///
    /// let str = bcoded_str.to_string();
    ///
    /// assert_eq!(&str, "hello");
    /// ```
    pub fn to_string(&self) -> String {
        let result = String::from_utf8(self.contents.clone());

        if result.is_err() {
            panic!("oxidation_bencode error: bytes in string are not ASCII or are incorrect");
        } else {
            result.unwrap()
        }
    }

    /// Creates a bencoded string from a string literal in bencoded format.
    ///
    /// # Examples
    ///
    /// ```
    /// use oxidation_bencode::string::BencodeString;
    ///
    /// let str_lit = "12:Hello World!";
    /// let ben_str = BencodeString::from_string(str_lit);
    /// let fin_str = ben_str.to_string();
    ///
    /// assert_eq!(ben_str.length, 12);
    /// assert_eq!(fin_str, "Hello World!");
    /// ```
    pub fn from_string(str: &'static str) -> BencodeString {
        let mut split_string = (&str).split(':');
        let pos_count = split_string.next().unwrap();
        let pos_contents = split_string.next().unwrap();
        if split_string.count() != 0 {
            panic!("oxidation_bencode error: string was encoded incorrectly and could not be converted");
        } else {
            let actual_count = i64::from_str_radix(pos_count, 10);
            if actual_count.is_err() {
                panic!("oxidation_bencode error: count was unable to be parsed correctly");
            } else {
                let final_count: i64 = actual_count.unwrap();
                let mut final_contents: Vec<u8> = Vec::new();

                for c in pos_contents.chars() {
                    final_contents.push(c as u8);
                }

                BencodeString {
                    length: final_count,
                    contents: final_contents.clone()
                }
            }
        }
    }
}