use super::Kernel;
use crate::{KernelAdd, KernelError, KernelMul};
use rayon::prelude::*;
use std::{error::Error, ops::Add, ops::Mul};

const PARAMS_LEN: usize = 2;

fn norm_pow(x: &Vec<f64>, xprime: &Vec<f64>) -> f64 {
    x.par_iter()
        .zip(xprime.par_iter())
        .map(|(x_i, xprime_i)| (x_i - xprime_i).powi(2))
        .sum()
}

#[derive(Clone, Debug)]
pub struct RBF;

impl Kernel<Vec<f64>> for RBF {
    fn params_len(&self) -> usize {
        PARAMS_LEN
    }

    fn value(
        &self,
        params: &[f64],
        x: &Vec<f64>,
        xprime: &Vec<f64>,
        with_grad: bool,
    ) -> Result<(f64, Vec<f64>), Box<dyn Error>> {
        if params.len() != PARAMS_LEN {
            return Err(KernelError::ParametersLengthMismatch.into());
        }
        if x.len() != xprime.len() {
            return Err(KernelError::InvalidArgument.into());
        }

        let norm_pow = norm_pow(x, xprime);

        let fx = params[0] * (-norm_pow / params[1]).exp();

        let gfx = if !with_grad {
            vec![]
        } else {
            let mut gfx = vec![f64::default(); PARAMS_LEN];

            gfx[0] = (-norm_pow / params[1]).exp();
            gfx[1] = params[0] * (-norm_pow / params[1]).exp() * (norm_pow / params[1].powi(2));

            gfx
        };

        Ok((fx, gfx))
    }
}

impl<R> Add<R> for RBF
where
    R: Kernel<Vec<f64>>,
{
    type Output = KernelAdd<Self, R, Vec<f64>>;

    fn add(self, rhs: R) -> Self::Output {
        Self::Output::new(self, rhs)
    }
}

impl<R> Mul<R> for RBF
where
    R: Kernel<Vec<f64>>,
{
    type Output = KernelMul<Self, R, Vec<f64>>;

    fn mul(self, rhs: R) -> Self::Output {
        Self::Output::new(self, rhs)
    }
}

#[cfg(test)]
mod tests {
    use crate::*;
    #[test]
    fn it_works() {
        let kernel = RBF;

        let (func, grad) = kernel
            .value(
                &[1.0, 1.0],
                &vec![1.0, 2.0, 3.0],
                &vec![3.0, 2.0, 1.0],
                true,
            )
            .unwrap();

        println!("{}", func);
        println!("{:#?}", grad);
    }
}
