#![allow(missing_docs)]

use super::*;

pub trait HasMinMax {
    fn min_value(self, other: Self) -> Self;
    fn max_value(self, other: Self) -> Self;
}

pub trait MiscNumeric<T: HasMinMax> {
    fn min_elementwise(self, value: T) -> Self;
    fn max_elementwise(self, value: T) -> Self;

    fn min_element(&self) -> Option<T>;
    fn max_element(&self) -> Option<T>;
}

impl<T: HasMinMax + Copy, N: ArrayLength<T>> MiscNumeric<T> for NumericArray<T, N> {
    #[inline(always)]
    fn min_elementwise(self, value: T) -> Self {
        NumericArray::from(self.0.map(|x| x.min_value(value)))
    }

    #[inline(always)]
    fn max_elementwise(self, value: T) -> Self {
        NumericArray::from(self.0.map(|x| x.max_value(value)))
    }

    #[inline]
    fn min_element(&self) -> Option<T> {
        self.0.first().map(|min| {
            let mut min = *min;

            self.0.iter().for_each(|x| {
                min = x.min_value(min);
            });

            min
        })
    }

    #[inline]
    fn max_element(&self) -> Option<T> {
        self.0.first().map(|max| {
            let mut max = *max;

            self.0.iter().for_each(|x| {
                max = x.max_value(max);
            });

            max
        })
    }
}

macro_rules! impl_has_minmax {
    ($($t:ty),*) => {
        $(
            impl HasMinMax for $t {
                #[inline(always)]
                fn min_value(self, other: Self) -> Self {
                    self.min(other)
                }

                #[inline(always)]
                fn max_value(self, other: Self) -> Self {
                    self.max(other)
                }
            }
        )*
    }
}

impl_has_minmax!(u8, u16, u32, u64, u128, usize, i8, i16, i32, i64, i128, isize, f32, f64);
