/* -*- Mode: C++; tab-width: 2; indent-tabs-mode: nil; c-basic-offset: 2 -*- */
/* This Source Code Form is subject to the terms of the Mozilla Public
 * License, v. 2.0. If a copy of the MPL was not distributed with this
 * file, You can obtain one at http://mozilla.org/MPL/2.0/. */

#include "mp4parse.h"

#include <assert.h>
#include <stdint.h>
#include <stdio.h>
#include <string.h>
#include <vector>

struct read_vector {
  explicit read_vector(FILE* file);
  size_t size() const;
  size_t location;
  std::vector<uint8_t> buffer;
};

read_vector::read_vector(FILE* file)
 : location(0)
{
  buffer.resize(4096);
  fprintf(stderr, "buffer resized to %llu\n",
      (unsigned long long)buffer.size());
  size_t read = fread(buffer.data(), sizeof(decltype(buffer)::value_type),
      buffer.size(), file);
  fprintf(stderr, "read %llu bytes from file %p\n",
      (unsigned long long)read, file);
  buffer.resize(read);
}

static intptr_t
vector_reader(uint8_t* buffer, uintptr_t size, void* userdata)
{
  if (!buffer || !userdata) {
    return -1;
  }
  auto source = static_cast<read_vector*>(userdata);
  if (source->location > source->buffer.size()) {
    return -1;
  }
  uintptr_t available = source->buffer.size() - source->location;
  uintptr_t length = std::min(available, size);
  memcpy(buffer, source->buffer.data() + source->location, length);
  source->location += length;
  fprintf(stderr, "read %llu bytes. position now %lld\n",
      (unsigned long long)length, (long long)source->location);
  return length;
}

int main() 
{
  FILE* f = fopen("street.mp4", "rb");
  assert(f != nullptr);
  read_vector reader = read_vector(f);
  fclose(f);

  mp4parse_io io = { vector_reader, &reader };
  mp4parse_parser* context = mp4parse_new(&io);
  assert(context != nullptr);

  int32_t rv = mp4parse_read(context);
  assert(rv == MP4PARSE_OK);
  fprintf(stderr, "mp4parse_read returned %d\n", rv);

  uint32_t tracks = mp4parse_get_track_count(context);
  assert(tracks == 2U);

  mp4parse_free(context);

  return 0;
}
#if 0
static intptr_t
file_reader(uint8_t* buffer, uintptr_t size, void* userdata)
{
  if (!buffer || !userdata) {
    return -1;
  }
  auto file = reinterpret_cast<FILE*>(userdata);
  auto read = fread(buffer, 1, size, file);
  if (read == 0 && feof(file)) {
    return 0;
  }
  if (ferror(file)) {
    return -1;
  }
  return read;
}

TEST(rust, MP4Metadata)
{
  FILE* f = fopen("street.mp4", "rb");
  ASSERT_TRUE(f != nullptr);
  mp4parse_io io = { file_reader, f };

  mp4parse_parser* context = mp4parse_new(&io);
  ASSERT_NE(context, nullptr);

  int32_t rv = mp4parse_read(context);
  EXPECT_EQ(rv, MP4PARSE_OK);
  fclose(f);

  uint32_t tracks = mp4parse_get_track_count(context);
  EXPECT_EQ(tracks, 2U);

  mp4parse_free(context);
}
#endif
