extern crate difference;
#[macro_use]
extern crate clap;

use clap::{App, Arg};
use std::fs::File;
use std::io;
use std::io::{BufRead, BufReader};
use std::path::Path;

use difference::Changeset;
use difference::Difference;
use difference::Difference::{Add, Rem, Same};

#[macro_use]
extern crate prettytable;
use prettytable::Table;

fn get_line_from_file(filename: &Path) -> String {
    if !filename.exists() || !filename.is_file() {
        println!("Cannot find file1: {:?}", filename);
        return "".to_string();
    }
    let file = File::open(filename).unwrap();
    let reader = BufReader::new(file);

    let mut s = "".to_owned();
    for (index, line) in reader.lines().enumerate() {
        let line = line.unwrap();

        if index == 0 {
            s = line.to_owned();
        } else {
            println!("File contains additional lines that will be ignored");
            break;
        }
    }
    s.to_string()
}

fn get_lines_from_file(filename: &Path) -> (String, String) {
    let file = File::open(filename).unwrap();
    let reader = BufReader::new(file);

    let mut s1 = "".to_owned();
    let mut s2 = "".to_owned();
    for (index, line) in reader.lines().enumerate() {
        let line = line.unwrap();

        if index == 0 {
            s1 = line.to_owned();
        } else if index == 1 {
            s2 = line.to_owned();
        } else {
            println!("File contains additional lines that will be ignored");
            break;
        }
    }
    (s1.to_string(), s2.to_string())
}

fn get_line_from_cmd(line_number: i32) -> String {
    println!("Please provide line #{}: ", line_number);
    let mut buffer = String::new();
    io::stdin().read_line(&mut buffer).expect("");
    buffer.trim().to_string()
}

fn get_line(line_number: i32, filepath: Option<&str>) -> String {
    match filepath {
        Some(filepath) => get_line_from_file(Path::new(filepath)),
        None => get_line_from_cmd(line_number),
    }
}

fn print_results(diffs: Vec<Difference>) {
    let mut table = Table::new();
    table.add_row(row!["Line 1", "Same", "Line 2"]);
    for d in diffs {
        match d {
            Same(line) => table.add_row(row!["", line, ""]),
            Add(line) => table.add_row(row!["", "", line]),
            Rem(line) => table.add_row(row![line, "", ""]),
        };
    }
    table.printstd();
}

fn preprocess_chunks(s: &str, separator: &[char], sort: bool) -> String {
    let mut chunks: Vec<&str> = s.split(|c| separator.contains(&c)).collect();
    if sort {
        chunks.sort();
    }
    chunks.join("\n")
}

fn main() {
    let matches = App::new("Line diff")
        .version(crate_version!())
        .author(crate_authors!())
        .about("Compare two lines by splitting the lines into smaller chunks and comparing the chunks. \
        There are multiple ways of specifying the two lines: \n \
        \ta single file that contains the two lines (--file option) \n \
        \tspecifying the two lines separately as a file path (indexed argument 1 and 2), on the command line (--line1 and --line2) or using command line input.")
        .arg(
            Arg::with_name("file")
                .help("A single file containing two lines. Remaining lines will be ignored.")
                .short("f")
                .takes_value(true),
        )
        .arg(
            Arg::with_name("file1")
                .help("Path to file containing the first line. Remaining lines will be ignored.")
                .takes_value(true),
        )
        .arg(
            Arg::with_name("file2")
                .help("Path to file containing the second line. Remaining lines will be ignored.")
                .takes_value(true),
        )
        .arg(
            Arg::with_name("line1")
                .long("line1")
                .help("First line as string")
                .takes_value(true),
        )
        .arg(
            Arg::with_name("line2")
                .long("line2")
                .help("Second line as string")
                .takes_value(true),
        )
        .arg(
            Arg::with_name("separator")
                .short("s")
                .help("Separator for splitting lines. It is possible to define multiple separators.")
                .takes_value(true)
                .multiple(true),
        )
        .arg(
            Arg::with_name("sorted")
                .short("o")
                .help("Whether or not the chunks should be sorted before comparing."),
        )
        .get_matches();

    let (s1, s2) = if let Some(filepath) = matches.value_of("file") {
        let path_file = Path::new(filepath);
        if !path_file.exists() || !path_file.is_file() {
            println!("Cannot find file1: {}", filepath);
            return;
        }
        get_lines_from_file(path_file)
    } else {
        let l1 = if let Some(l1) = matches.value_of("line1") {
            l1.to_owned()
        } else {
            get_line(1, matches.value_of("file1"))
        };
        let l2 = if let Some(l2) = matches.value_of("line2") {
            l2.to_owned()
        } else {
            get_line(2, matches.value_of("file2"))
        };
        (l1, l2)
    };

    let sort = matches.is_present("sorted");

    let separator_chars = if matches.is_present("separator") {
        let separators = matches.values_of("separator").unwrap().collect::<Vec<_>>();
        let mut separator_chars: Vec<char> = Vec::new();
        for s in separators {
            println!("Separator: '{}'", s);
            for character in s.chars() {
                separator_chars.push(character);
            }
        }
        separator_chars
    } else {
        vec![' ']
    };
    println!("Line 1: \n{}", s1);
    println!("Line 2: \n{}", s2);

    let s1 = preprocess_chunks(&s1, &separator_chars, sort);
    let s2 = preprocess_chunks(&s2, &separator_chars, sort);

    let changeset = Changeset::new(&s1, &s2, "\n");
    print_results(changeset.diffs);
}

#[cfg(test)]
mod tests {
    use super::*;
    #[test]
    fn preprocess_no_sorting() {
        let output = preprocess_chunks("hello world", &vec![' '], false);
        assert_eq!("hello\nworld", output);
        let output = preprocess_chunks("hello world", &vec![';'], false);
        assert_eq!("hello world", output);
        let output = preprocess_chunks("hello world", &vec!['o'], false);
        assert_eq!("hell\n w\nrld", output);
    }

    #[test]
    fn preprocess_sorting() {
        let output = preprocess_chunks("a b c", &vec![' '], true);
        assert_eq!("a\nb\nc", output);
        let output = preprocess_chunks("c b a", &vec![' '], true);
        assert_eq!("a\nb\nc", output);
    }

    #[test]
    fn preprocess_multiple_separators() {
        let output = preprocess_chunks("a b;c", &vec![' '], true);
        assert_eq!("a\nb;c", output);
        let output = preprocess_chunks("c b a", &vec![' ', ';'], true);
        assert_eq!("a\nb\nc", output);
    }

    #[test]
    fn read_one_line() {
        let l1 = get_line_from_file(Path::new("test.txt"));
        assert_eq!("Hello world 1 3 .", l1);
    }

    #[test]
    fn read_two_lines() {
        let (l1, l2) = get_lines_from_file(Path::new("test.txt"));
        assert_eq!("Hello world 1 3 .", l1);
        assert_eq!("as the %+3^ night", l2);
    }
}
