// Copyright 2016 Nathan Sizemore <nathanrsizemore@gmail.com>
//
// This Source Code Form is subject to the terms of the
// Mozilla Public License, v. 2.0. If a copy of the MPL was not
// distributed with this file, You can obtain one at
// http://mozilla.org/MPL/2.0/.


#[macro_use]
extern crate bitflags;
extern crate libc;
extern crate errno;


use std::mem;
use std::ffi::CString;
use std::os::unix::io::{RawFd, AsRawFd};
use std::io::{Read, Write, Error, ErrorKind};

use libc::{c_int, c_void};
use errno::errno;


bitflags! {
    pub flags SocketType: libc::c_int {
        const SOCK_STREAM   = libc::SOCK_STREAM,
        const SOCK_DGRAM    = libc::SOCK_DGRAM,
        const SOCK_RAW      = libc::SOCK_RAW,
        const SOCK_CLOEXEC  = libc::SOCK_CLOEXEC,
        const SOCK_NONBLOCK = libc::SOCK_NONBLOCK
    }
}

bitflags! {
    pub flags IoFlags: libc::c_int {
        const MSG_CMSG_DONTWAIT = 0x0040,
        const MSG_ERRQUEUE      = 0x2000,
        const MSG_OOB           = 0x0001,
        const MSG_PEEK          = 0x0002,
        const MSG_TRUNC         = 0x0020,
        const MSG_WAITALL       = 0x0100,
        const MSG_CONFIRM       = 0x0800,
        const MSG_DONTROUTE     = 0x0004,
        const MSG_EOR           = 0x0080,
        const MSG_MORE          = 0x8000,
        const MSG_NOSIGNAL      = 0x4000
    }
}

pub enum Domain {
    Unix,
    Inet,
    Inet6,
    Netlink,
    Packet
}

pub enum Shutdown {
    Read,
    Write,
    Both
}

pub struct Socket {
    fd: c_int
}

impl Domain {
    fn to_cint(&self) -> c_int {
        match *self {
            Domain::Unix => libc::AF_UNIX,
            Domain::Inet => libc::AF_INET,
            Domain::Inet6 => libc::AF_INET6,
            Domain::Netlink => libc::AF_NETLINK,
            Domain::Packet => libc::AF_PACKET,
        }
    }
}

impl Shutdown {
    fn to_cint(&self) -> c_int {
        match *self {
            Shutdown::Read => libc::SHUT_RD,
            Shutdown::Write => libc::SHUT_WR,
            Shutdown::Both => libc::SHUT_RDWR
        }
    }
}

impl Socket {
    /// Creates a new socket with the passed options.
    pub fn new(domain: Domain,
               sock_type: c_int,
               protocol: c_int)
               -> Result<Socket, Error>
    {
        let result = unsafe {
            libc::socket(domain.to_cint(), sock_type, protocol)
        };

        if result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        return Ok(Socket {
            fd: result
        });
    }

    /// Creates a socket from an already established socket fd.
    pub fn from_fd(fd: c_int) -> Socket {
        return Socket {
            fd: fd
        };
    }

    /// Assigns the address specified by `address` to the socket
    pub fn bind(&self,
                address: *const libc::sockaddr,
                address_len: libc::socklen_t)
                -> Result<(), Error>
    {
        let result = unsafe {
            libc::bind(self.fd, address, address_len)
        };

        if result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        return Ok(());
    }

    /// Connects the socket to `address`.
    pub fn connect(&self,
                   address: *const libc::sockaddr,
                   address_len: libc::socklen_t)
                   -> Result<(), Error>
    {
        let result = unsafe {
            libc::connect(self.fd, address, address_len)
        };

        if result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        return Ok(());
    }

    /// Marks the socket as a socket that will be used to accept incoming connections.
    pub fn listen(&self, backlog: usize) -> Result<(), Error> {
        let result = unsafe {
            libc::listen(self.fd, backlog as c_int)
        };

        if result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        return Ok(());
    }

    /// Extracts the first connection from the backlog queue of pending connections and creates
    /// and returns a new `Socket`.
    pub fn accept(&self,
                  address: *mut libc::sockaddr,
                  address_len: *mut libc::socklen_t)
                  -> Result<Socket, Error>
    {
        let result = unsafe {
            libc::accept(self.fd, address, address_len)
        };

        if result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        return Ok(Socket{
            fd: result
        });
    }

    /// If no messages are available at the socket, the call waits for a message to arrive, unless
    /// the socket is nonblocking, in which case `ErrorKind::WouldBlock` is returned. The call
    /// normally returns any data available, up to the requested amount, rather than waiting for
    /// receipt of the full amount requested.
    pub fn recv(&self, buf: &mut [u8], flags: libc::c_int) -> Result<usize, Error> {
        let len = buf.len() as libc::size_t;
        let buf_ptr = buf.as_mut_ptr() as *mut _ as *mut libc::c_void;
        
        let result = unsafe {
            libc::recv(self.fd, buf_ptr, len, flags)
        };

        if result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        if result == 0 {
            return Err(Error::new(ErrorKind::UnexpectedEof, "Unexpected end of file"))
        }

        let num_read = result as usize;
        return Ok(num_read);
    }

    /// If the message is too long to pass atomically through the underlying protocol, the error
    /// `EMSGSIZE` is returned, and the message is not transmitted. No indication of failure
    /// to deliver is implicit in a send(). Locally detected errors are indicated by an Error
    /// returned. When the message does not fit into the send buffer of the socket, send()
    /// normally blocks, unless the socket has been placed in nonblocking I/O mode.
    /// In nonblocking mode it would fail with the error `ErrorKind::WouldBlock` in this case.
    pub fn send(&self, buf: &[u8], flags: libc::c_int) -> Result<usize, Error> {
        let len = buf.len() as libc::size_t;
        let buf_ptr = buf.as_ptr() as *const _ as *const libc::c_void;

        let result = unsafe {
            libc::send(self.fd, buf_ptr, len, flags)
        };

        if result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        if result == 0 && len > 0 {
            return Err(Error::new(ErrorKind::Other, "Send returned zero (0)"));
        }

        let num_sent = result as usize;
        return Ok(num_sent);
    }

    /// Disables subsequent send and/or receive operations on a socket, depending on the value
    /// of the how argument.
    pub fn shutdown(&self, how: Shutdown) -> Result<(), Error> {
        let result = unsafe {
            libc::shutdown(self.fd, how.to_cint())
        };

        if result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        return Ok(());
    }

    /// Closes the underlying file descriptor of the socket.
    pub fn close(&self) -> Result<(), Error> {
        let result = unsafe {
            libc::close(self.fd)
        };

        if result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        return Ok(());
    }

    /// If set, disable the Nagle algorithm. This means that segments are always sent as soon as
    /// possible, even if there is only a small amount of data. When not set, data is buffered
    /// until there is a sufficient amount to send out, thereby avoiding the frequent sending of
    /// small packets, which results in poor utilization of the network. This option is
    /// overridden by TCP_CORK; however, setting this option forces an explicit flush of pending
    /// output, even if TCP_CORK is currently set.
    pub fn set_tcp_nodelay(&mut self, nodelay: bool) -> Result<(), Error> {
        let optval: c_int = match nodelay {
            true => 1,
            false => 0
        };
        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::IPPROTO_TCP,
                             libc::TCP_NODELAY,
                             &optval as *const _ as *const c_void,
                             mem::size_of::<c_int>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Sets the `O_NONBLOCK` flag on the underlying fd
    pub fn set_nonblocking(&mut self) -> Result<(), Error> {
        let result = unsafe {
            libc::fcntl(self.as_raw_fd(), libc::F_GETFL, 0)
        };
        if result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        let flags = result | libc::O_NONBLOCK;
        let result = unsafe {
            libc::fcntl(self.as_raw_fd(), libc::F_SETFL, flags)
        };
        if result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Bind this socket to a particular device like "eth0", as specified in the passed
    /// interface name. If the name is an empty string or the option length is zero, the socket
    /// device binding is removed. The passed option is a variable-length null-terminated
    /// interface name string with the maximum size of IFNAMSIZ. If a socket is bound to an
    /// interface, only packets received from that particular interface are processed by the
    /// socket. Note that this only works for some socket types, particularly AF_INET sockets.
    /// It is not supported for packet sockets (use normal bind(2) there).
    ///
    /// Before Linux 3.8, this socket option could be set, but could not retrieved with
    /// getsockopt(2). Since Linux 3.8, it is readable. The optlen argument should contain the
    /// buffer size available to receive the device name and is recommended to be IFNAMSZ bytes.
    /// The real device name length is reported back in the optlen argument.
    pub fn set_bindtodevice(&mut self, interface: String) -> Result<(), Error> {
        const SO_BINDTODEVICE: i32 = 25;
        let cstr_result = CString::new(interface);
        if cstr_result.is_err() {
            return Err(Error::new(ErrorKind::Other, "Null Byte"));
        }

        let cstr = cstr_result.unwrap();
        unsafe {
            if libc::strlen(cstr.as_ptr()) > libc::IF_NAMESIZE {
                return Err(Error::new(ErrorKind::Other, "strlen(interface) > IFNAMSIZ"));
            }
        }

        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             SO_BINDTODEVICE,
                             cstr.as_ptr() as *const c_void,
                             libc::strlen(cstr.as_ptr()) as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// When enabled, datagram sockets are allowed to send packets to a broadcast address.
    /// This option has no effect on stream-oriented sockets.
    pub fn set_broadcast(&mut self, option: bool) -> Result<(), Error> {
        let optval: c_int = match option {
            true => 1,
            false => 0
        };
        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             libc::SO_BROADCAST,
                             &optval as *const _ as *const c_void,
                             mem::size_of::<c_int>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Enable BSD bug-to-bug compatibility. This is used by the UDP protocol module in
    /// Linux 2.0 and 2.2. If enabled ICMP errors received for a UDP socket will not be passed
    /// to the user program. In later kernel versions, support for this option has been phased
    /// out: Linux 2.4 silently ignores it, and Linux 2.6 generates a kernel warning
    /// (printk()) if a program uses this option. Linux 2.0 also enabled BSD bug-to-bug
    /// compatibility options (random header changing, skipping of the broadcast flag) for raw
    /// sockets with this option, but that was removed in Linux 2.2.
    pub fn set_bsdcompat(&mut self, option: bool) -> Result<(), Error> {
        const SO_BSDCOMPAT: i32 = 14;
        let optval: c_int = match option {
            true => 1,
            false => 0
        };
        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             SO_BSDCOMPAT,
                             &optval as *const _ as *const c_void,
                             mem::size_of::<c_int>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Enable socket debugging. Only allowed for processes with the CAP_NET_ADMIN capability
    /// or an effective user ID of 0.
    pub fn set_debug(&mut self, option: bool) -> Result<(), Error> {
        let optval: c_int = match option {
            true => 1,
            false => 0
        };
        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             libc::SO_DEBUG,
                             &optval as *const _ as *const c_void,
                             mem::size_of::<c_int>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Don't send via a gateway, only send to directly connected hosts. The same effect can be
    /// achieved by setting the MSG_DONTROUTE flag on a socket send(2) operation. Expects an
    /// integer boolean flag.
    pub fn set_dontroute(&mut self, option: bool) -> Result<(), Error> {
        let optval: c_int = match option {
            true => 1,
            false => 0
        };
        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             libc::SO_DONTROUTE,
                             &optval as *const _ as *const c_void,
                             mem::size_of::<c_int>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Enable sending of keep-alive messages on connection-oriented sockets. Expects an integer
    /// boolean flag.
    pub fn set_keepalive(&mut self, option: bool) -> Result<(), Error> {
        let optval: c_int = match option {
            true => 1,
            false => 0
        };
        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             libc::SO_KEEPALIVE,
                             &optval as *const _ as *const c_void,
                             mem::size_of::<c_int>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Sets or gets the SO_LINGER option. When enabled, a close(2) or shutdown(2) will not return
    /// until all queued messages for the socket have been successfully sent or the linger timeout
    /// has been reached. Otherwise, the call returns immediately and the closing is done in the
    /// background. When the socket is closed as part of exit(2), it always lingers in the
    /// background.
    pub fn set_linger(&mut self, option: bool, sec: u32) -> Result<(), Error> {
        #[repr(C, packed)]
        struct Linger {
            l_onoff: c_int,
            l_linger: c_int
        };

        let optval: c_int = match option {
            true => 1,
            false => 0
        };
        let data = Linger {
            l_onoff: optval,
            l_linger: sec as i32
        };

        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             libc::SO_LINGER,
                             &data as *const _ as *const c_void,
                             mem::size_of::<Linger>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Set the mark for each packet sent through this socket (similar to the netfilter MARK
    /// target but socket-based). Changing the mark can be used for mark-based routing without
    /// netfilter or for packet filtering. Setting this option requires the CAP_NET_ADMIN
    /// capability.
    pub fn set_mark(&mut self, option: bool) -> Result<(), Error> {
        const SO_MARK: i32 = 36;
        let optval: c_int = match option {
            true => 1,
            false => 0
        };
        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             SO_MARK,
                             &optval as *const _ as *const c_void,
                             mem::size_of::<c_int>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// If this option is enabled, out-of-band data is directly placed into the receive data
    /// stream. Otherwise out-of-band data is only passed when the MSG_OOB flag is set during
    /// receiving.
    pub fn set_oobinline(&mut self, option: bool) -> Result<(), Error> {
        let optval: c_int = match option {
            true => 1,
            false => 0
        };
        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             libc::SO_OOBINLINE,
                             &optval as *const _ as *const c_void,
                             mem::size_of::<c_int>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Enable or disable the receiving of the SCM_CREDENTIALS control message. For more
    /// information see unix(7).
    pub fn set_passcred(&mut self, option: bool) -> Result<(), Error> {
        const SO_PASSCRED: i32 = 16;
        let optval: c_int = match option {
            true => 1,
            false => 0
        };
        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             SO_PASSCRED,
                             &optval as *const _ as *const c_void,
                             mem::size_of::<c_int>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Set the protocol-defined priority for all packets to be sent on this socket. Linux uses
    /// this value to order the networking queues: packets with a higher priority may be processed
    /// first depending on the selected device queueing discipline. For ip(7), this also sets the
    /// IP type-of-service (TOS) field for outgoing packets. Setting a priority outside the
    /// range 0 to 6 requires the CAP_NET_ADMIN capability.
    pub fn set_priority(&mut self, priority: u32) -> Result<(), Error> {
        const SO_PRIORITY: i32 = 12;
        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             SO_PRIORITY,
                             &priority as *const _ as *const c_void,
                             mem::size_of::<u32>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Sets or gets the maximum socket receive buffer in bytes. The kernel doubles this value
    /// (to allow space for bookkeeping overhead) when it is set using setsockopt(2), and this
    /// doubled value is returned by getsockopt(2). The default value is set by
    /// the /proc/sys/net/core/rmem_default file, and the maximum allowed value is set by
    /// the /proc/sys/net/core/rmem_max file. The minimum (doubled) value for this option is 256.
    pub fn set_rcvbuf(&mut self, size: usize) -> Result<(), Error> {
        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             libc::SO_RCVBUF,
                             &size as *const _ as *const c_void,
                             mem::size_of::<usize>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Using this socket option, a privileged (CAP_NET_ADMIN) process can perform the same task
    /// as SO_RCVBUF, but the rmem_max limit can be overridden.
    pub fn set_rcvbufforce(&mut self, size: usize) -> Result<(), Error> {
        self.set_rcvbuf(size)
    }

    /// Specify the minimum number of bytes in the buffer until the socket layer will pass the
    /// data to the protocol (SO_SNDLOWAT) or the user on receiving (SO_RCVLOWAT). These two
    /// values are initialized to 1. SO_SNDLOWAT is not changeable on Linux (setsockopt(2) fails
    /// with the error ENOPROTOOPT). SO_RCVLOWAT is changeable only since Linux 2.4. The select(2)
    /// and poll(2) system calls currently do not respect the SO_RCVLOWAT setting on Linux, and
    /// mark a socket readable when even a single byte of data is available. A subsequent read
    /// from the socket will block until SO_RCVLOWAT bytes are available.
    pub fn set_rcvlowat(&mut self, bytes: usize) -> Result<(), Error> {
        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             libc::SO_RCVLOWAT,
                             &bytes as *const _ as *const c_void,
                             mem::size_of::<usize>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Specify the minimum number of bytes in the buffer until the socket layer will pass the
    /// data to the protocol (SO_SNDLOWAT) or the user on receiving (SO_RCVLOWAT). These two
    /// values are initialized to 1. SO_SNDLOWAT is not changeable on Linux (setsockopt(2) fails
    /// with the error ENOPROTOOPT). SO_RCVLOWAT is changeable only since Linux 2.4. The select(2)
    /// and poll(2) system calls currently do not respect the SO_RCVLOWAT setting on Linux, and
    /// mark a socket readable when even a single byte of data is available. A subsequent read
    /// from the socket will block until SO_RCVLOWAT bytes are available.
    pub fn set_sndlowat(&mut self, bytes: usize) -> Result<(), Error> {
        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             libc::SO_SNDLOWAT,
                             &bytes as *const _ as *const c_void,
                             mem::size_of::<usize>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Specify the receiving or sending timeouts until reporting an error. The argument is a
    /// struct timeval. If an input or output function blocks for this period of time, and data
    /// has been sent or received, the return value of that function will be the amount of data
    /// transferred; if no data has been transferred and the timeout has been reached then -1 is
    /// returned with errno set to EAGAIN or EWOULDBLOCK, or EINPROGRESS (for connect(2)) just as
    /// if the socket was specified to be nonblocking. If the timeout is set to zero (the default)
    /// then the operation will never timeout. Timeouts only have effect for system calls that
    /// perform socket I/O (e.g., read(2), recvmsg(2), send(2), sendmsg(2)); timeouts have no
    /// effect for select(2), poll(2), epoll_wait(2), and so on.
    pub fn set_rcvtimeo(&mut self,
                        sec: libc::time_t,
                        micro_sec: libc::suseconds_t)
                        -> Result<(), Error>
    {
        #[repr(C, packed)]
        struct Timeval {
            tv_sec: libc::time_t,
            tv_usec: libc::suseconds_t
        };
        let data = Timeval {
            tv_sec: sec,
            tv_usec: micro_sec
        };

        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             libc::SO_RCVTIMEO,
                             &data as *const _ as *const c_void,
                             mem::size_of::<Timeval>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Specify the receiving or sending timeouts until reporting an error. The argument is a
    /// struct timeval. If an input or output function blocks for this period of time, and data
    /// has been sent or received, the return value of that function will be the amount of data
    /// transferred; if no data has been transferred and the timeout has been reached then -1 is
    /// returned with errno set to EAGAIN or EWOULDBLOCK, or EINPROGRESS (for connect(2)) just as
    /// if the socket was specified to be nonblocking. If the timeout is set to zero (the default)
    /// then the operation will never timeout. Timeouts only have effect for system calls that
    /// perform socket I/O (e.g., read(2), recvmsg(2), send(2), sendmsg(2)); timeouts have no
    /// effect for select(2), poll(2), epoll_wait(2), and so on.
    pub fn set_sndtimeo(&mut self,
                        sec: libc::time_t,
                        micro_sec: libc::suseconds_t)
                        -> Result<(), Error>
    {
        #[repr(C, packed)]
        struct Timeval {
            tv_sec: libc::time_t,
            tv_usec: libc::suseconds_t
        };
        let data = Timeval {
            tv_sec: sec,
            tv_usec: micro_sec
        };

        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             libc::SO_SNDTIMEO,
                             &data as *const _ as *const c_void,
                             mem::size_of::<Timeval>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Indicates that the rules used in validating addresses supplied in a bind(2) call should
    /// allow reuse of local addresses. For AF_INET sockets this means that a socket may bind,
    /// except when there is an active listening socket bound to the address. When the listening
    /// socket is bound to INADDR_ANY with a specific port then it is not possible to bind to this
    /// port for any local address. Argument is an integer boolean flag.
    pub fn set_reuseaddr(&mut self, option: bool) -> Result<(), Error> {
        let optval: c_int = match option {
            true => 1,
            false => 0
        };
        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             libc::SO_REUSEADDR,
                             &optval as *const _ as *const c_void,
                             mem::size_of::<c_int>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Sets or gets the maximum socket send buffer in bytes. The kernel doubles this value
    /// (to allow space for bookkeeping overhead) when it is set using setsockopt(2), and this
    /// doubled value is returned by getsockopt(2). The default value is set by
    /// the /proc/sys/net/core/wmem_default file and the maximum allowed value is set by
    /// the /proc/sys/net/core/wmem_max file. The minimum (doubled) value for this option is 2048.
    pub fn set_sndbuf(&mut self, size: usize) -> Result<(), Error> {
        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             libc::SO_SNDBUF,
                             &size as *const _ as *const c_void,
                             mem::size_of::<usize>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }

    /// Using this socket option, a privileged (CAP_NET_ADMIN) process can perform the same task
    /// as SO_SNDBUF, but the wmem_max limit can be overridden.
    pub fn set_sndbufforce(&mut self, size: usize) -> Result<(), Error> {
        self.set_sndbuf(size)
    }

    /// Enable or disable the receiving of the SO_TIMESTAMP control message. The timestamp control
    /// message is sent with level SOL_SOCKET and the cmsg_data field is a struct timeval
    /// indicating the reception time of the last packet passed to the user in this call.
    /// See cmsg(3) for details on control messages.
    pub fn set_timestamp(&mut self, option: bool) -> Result<(), Error> {
        const SO_TIMESTAMP: i32 = 29;
        let optval: c_int = match option {
            true => 1,
            false => 0
        };
        let opt_result = unsafe {
            libc::setsockopt(self.fd,
                             libc::SOL_SOCKET,
                             SO_TIMESTAMP,
                             &optval as *const _ as *const c_void,
                             mem::size_of::<c_int>() as u32)
        };
        if opt_result < 0 {
            return Err(Error::from_raw_os_error(errno().0 as i32));
        }

        Ok(())
    }
}

impl Read for Socket {
    fn read(&mut self, buf: &mut [u8]) -> Result<usize, Error> {
        return self.recv(buf, 0);
    }
}

impl Write for Socket {
    fn write(&mut self, buf: &[u8]) -> Result<usize, Error> {
        return self.send(buf, 0);
    }
    fn flush(&mut self) -> Result<(), Error> {
        return Ok(());
    }
}

impl AsRawFd for Socket {
    fn as_raw_fd(&self) -> RawFd {
        self.fd
    }
}

unsafe impl Send for Socket {}
unsafe impl Sync for Socket {}
