//! Builder patterns to manipulate an unprocessed thread of patches

use crate::Patch;

/// Abstraction over a generic email
#[derive(Clone, Debug, Eq, PartialEq)]
pub enum Mail<'mail> {
  /// A git-sent email with patch information
  Git(Patch<'mail>),
  /// A non git-mail reply
  Mail(String),
}

/// A tree of patches that can be filtered to yield a `PatchSet`
pub struct PatchTree<'mail> {
  /// The current patch
  pub this: Mail<'mail>,
  /// All replies to this patch mail
  pub children: Vec<PatchTree<'mail>>,
  /// Mark an email as selected
  pub selected: bool,
}

impl<'mail> PatchTree<'mail> {
  pub fn new(initial: Patch<'mail>) -> Self {
    Self {
      this: Mail::Git(initial),
      children: Default::default(),
      selected: false,
    }
  }

  /// Select a specific mail into the tree
  pub fn select(&mut self) {
    self.selected = true;
  }

  /// Deselect a single mail from the tree
  pub fn deselect(&mut self) {
    self.selected = false;
  }
}
