use lib_ruby_parser_nodes::{Message, Node};

/// A struct with codegen options
#[derive(Clone)]
pub struct Options {
    /// Identifier that is passed to wrapping `#ifndef`
    ///
    /// Default: `LIB_RUBY_PARSER_BINDINGS_H`
    pub ifndef_name: &'static str,

    /// Pre-code
    ///
    /// You can put here things like `#include` or
    /// `#ifdef __cplusplus \n extern "C" ...`
    ///
    /// Default: `"// pre-code"`
    pub pre_code: &'static str,

    /// Post-code
    ///
    /// Can be used to close `#ifdef __cplusplus`
    ///
    /// Default: `"// post-code"`
    pub post_code: &'static str,

    /// Attributes that are attached to every generated function
    ///
    /// can be `__attribute__((always_inline))` if you want to perform inlining
    pub fn_attributes: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `Ptr<T>` struct in `lib-ruby-parser`
    pub ptr_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `MaybePtr<T>` struct in `lib-ruby-parser`
    pub maybe_ptr_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `&'static strPtr` struct in `lib-ruby-parser`
    pub string_ptr_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `Maybe&'static strPtr` struct in `lib-ruby-parser`
    pub maybe_string_ptr_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `SharedByteList` struct in `lib-ruby-parser`
    pub shared_byte_list_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `u8` struct in `lib-ruby-parser`
    ///
    /// In most cases if you build low-level bindings it will be u8
    /// (without converting it to/from blob)
    pub byte_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `Token` struct in `lib-ruby-parser`
    pub token_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `Node` struct in `lib-ruby-parser`
    pub node_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `Diagnostic` struct in `lib-ruby-parser`
    pub diagnostic_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `CommentType` struct in `lib-ruby-parser`
    pub comment_type_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `Comment` struct in `lib-ruby-parser`
    pub comment_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `MagicCommentKind` struct in `lib-ruby-parser`
    pub magic_comment_kind_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `MagicComment` struct in `lib-ruby-parser`
    pub magic_comment_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `Bytes` struct in `lib-ruby-parser`
    pub bytes_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `Loc` struct in `lib-ruby-parser`
    pub loc_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `MaybeLoc` struct in `lib-ruby-parser`
    pub maybe_loc_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `ErrorLevel` struct in `lib-ruby-parser`
    pub error_level_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `Diagnostic` struct in `lib-ruby-parser`
    pub diagnostic_message_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `SourceLine` struct in `lib-ruby-parser`
    pub source_line_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `List<Byte>` struct in `lib-ruby-parser`
    pub byte_list_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `List<Token>` struct in `lib-ruby-parser`
    pub token_list_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `List<Node>` struct in `lib-ruby-parser`
    pub node_list_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `List<Diagnostic>` struct in `lib-ruby-parser`
    pub diagnostic_list_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `List<Comment>` struct in `lib-ruby-parser`
    pub comment_list_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `List<MagicComment>` struct in `lib-ruby-parser`
    pub magic_comment_list_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `List<SourceLine>` struct in `lib-ruby-parser`
    pub source_line_list_blob_name: &'static str,

    /// Function that takes a `Message` from `lib-ruby-parser-nodes`
    /// and returns a name of what you want to be used
    /// **as a blob** of its representation
    /// in in `lib-ruby-parser`
    ///
    /// For example, for `UnexpectedToken` message you may want to
    /// call your blob class/struct in C++ as `UnexpectedToken_BLOB`
    ///
    /// Default: fn that returns `<message.camelcase_name>_BLOB`
    pub message_variant_blob_name_fn: fn(&Message) -> String,

    /// Function that takes a `Node` from `lib-ruby-parser-nodes`
    /// and returns a name of what you want to be used
    /// **as a blob** of its representation
    /// in in `lib-ruby-parser`
    ///
    /// For example, for `Arg` message you may want to
    /// call your blob class/struct in C++ as `Arg_BLOB`
    ///
    /// Default: fn that returns `<node.camelcase_name>_BLOB`
    pub node_variant_blob_name_fn: fn(&Node) -> String,

    /// Name of what you want to be used
    /// **as a blob** of the `InputError` struct in `lib-ruby-parser`
    pub input_error_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `DecoderResult` struct in `lib-ruby-parser`
    pub decoder_result_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `Decoder` struct in `lib-ruby-parser`
    pub decoder_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `RewriteAction` struct in `lib-ruby-parser`
    pub rewrite_action_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `LexStateAction` struct in `lib-ruby-parser`
    pub lex_state_action_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `TokenRewriterResult` struct in `lib-ruby-parser`
    pub token_rewriter_result_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `TokenRewriter` struct in `lib-ruby-parser`
    pub token_rewriter_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `MaybeDecoder` struct in `lib-ruby-parser`
    pub maybe_decoder_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `MaybeTokenRewriter` struct in `lib-ruby-parser`
    pub maybe_token_rewriter_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `ParserOptions` struct in `lib-ruby-parser`
    pub parser_options_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `DecodedInput` struct in `lib-ruby-parser`
    pub decoded_input_blob_name: &'static str,

    /// Name of what you want to be used
    /// **as a blob** of the `ParserResult` struct in `lib-ruby-parser`
    pub parser_result_blob_name: &'static str,
}

fn default_message_variant_blob_name(message: &Message) -> String {
    format!("{}_BLOB", message.camelcase_name)
}

fn default_node_variant_blob_name(node: &Node) -> String {
    format!("{}_BLOB", node.camelcase_name)
}

impl Default for Options {
    fn default() -> Self {
        Options {
            ifndef_name: "LIB_RUBY_PARSER_BINDINGS_H",
            pre_code: "// pre-code",
            post_code: "// post-code",
            fn_attributes: "",

            // blob names
            ptr_blob_name: "Ptr_BLOB",
            maybe_ptr_blob_name: "MaybePtr_BLOB",
            string_ptr_blob_name: "StringPtr_BLOB",
            maybe_string_ptr_blob_name: "MaybeStringPtr_BLOB",
            shared_byte_list_blob_name: "SharedByteList_BLOB",
            byte_blob_name: "Byte_BLOB",
            token_blob_name: "Token_BLOB",
            node_blob_name: "Node_BLOB",
            diagnostic_blob_name: "Diagnostic_BLOB",
            comment_type_blob_name: "CommentType_BLOB",
            comment_blob_name: "Comment_BLOB",
            magic_comment_kind_blob_name: "MagicCommentKind_BLOB",
            magic_comment_blob_name: "MagicComment_BLOB",
            source_line_blob_name: "SourceLine_BLOB",
            bytes_blob_name: "Bytes_BLOB",
            loc_blob_name: "Loc_BLOB",
            maybe_loc_blob_name: "MaybeLoc_BLOB",
            error_level_blob_name: "ErrorLevel_BLOB",
            diagnostic_message_blob_name: "DiagnosticMessage_BLOB",

            // list blob names
            byte_list_blob_name: "ByteList_BLOB",
            token_list_blob_name: "TokenList_BLOB",
            node_list_blob_name: "NodeList_BLOB",
            diagnostic_list_blob_name: "DiagnosticList_BLOB",
            comment_list_blob_name: "CommentList_BLOB",
            magic_comment_list_blob_name: "MagicCommentList_BLOB",
            source_line_list_blob_name: "SourceLineList_BLOB",

            // messages
            message_variant_blob_name_fn: default_message_variant_blob_name,

            // nodes
            node_variant_blob_name_fn: default_node_variant_blob_name,

            input_error_blob_name: "InputError_BLOB",
            decoder_result_blob_name: "DecoderResult_BLOB",
            decoder_blob_name: "Decoder_BLOB",

            rewrite_action_blob_name: "RewriteAction_BLOB",
            lex_state_action_blob_name: "LexStateAction_BLOB",
            token_rewriter_result_blob_name: "TokenRewriterResult_BLOB",
            token_rewriter_blob_name: "TokenRewriter_BLOB",

            maybe_decoder_blob_name: "MaybeDecoder_BLOB",
            maybe_token_rewriter_blob_name: "MaybeTokenRewriter_BLOB",
            parser_options_blob_name: "ParserOptions_BLOB",

            decoded_input_blob_name: "DecodedInput_BLOB",
            parser_result_blob_name: "ParserResult_BLOB",
        }
    }
}

impl Options {
    /// Applies `self.message_variant_blob_name_fn` on a given `message`
    ///
    /// Utility function
    pub fn message_variant_blob_name(&self, message: &Message) -> String {
        (self.message_variant_blob_name_fn)(message)
    }

    /// Applies `self.node_variant_blob_name` on a given `node`
    ///
    /// Utility function
    pub fn node_variant_blob_name(&self, node: &Node) -> String {
        (self.node_variant_blob_name_fn)(node)
    }
}
