use crate::attribute::{
    AllowedType, ArrayItems, Attribute, ConstValue, DefinitionType, EnumValues, NumericMax,
    NumericMin, ObjectProperties, PointerPath, StringMaxLength, StringMinLength, StringPattern,
};
use crate::definition::{Definition, Type};
use crate::error::{Error, ValidationError};

use std::collections::{HashMap, HashSet};

use serde_json;

type NewAttribute =
    fn(state: &mut State, path: DocumentPath, ctx: &Context) -> Result<Box<Attribute>, Error>;

#[derive(Clone, Debug)]
pub struct DocumentPath {
    document_id: String,
    path: Vec<String>,
}

impl DocumentPath {
    pub fn new(document_id: &str) -> Self {
        DocumentPath {
            document_id: document_id.to_string(),
            path: vec![],
        }
    }

    pub fn add(&mut self, key: &str) {
        self.path.push(key.to_string());
    }

    pub fn path(&self) -> &Vec<String> {
        &self.path
    }

    pub fn document_id(&self) -> String {
        self.document_id.clone()
    }
}

pub struct State {
    types: HashMap<Type, HashMap<String, NewAttribute>>,
    required_attrs: HashMap<Type, HashSet<String>>,
    defs: HashMap<String, Definition>,
    unresolved_ptrs: HashMap<String, HashSet<String>>,
}

impl State {
    pub fn new() -> Result<Self, Error> {
        let mut state = State {
            types: HashMap::new(),
            required_attrs: HashMap::new(),
            unresolved_ptrs: HashMap::new(),
            defs: HashMap::new(),
        };

        // General
        state.add_attribute(
            "type",
            DefinitionType::build,
            DefinitionType::allowed_types(),
        )?;

        // String
        state.add_attribute(
            "minLength",
            StringMinLength::build,
            StringMinLength::allowed_types(),
        )?;
        state.add_attribute(
            "maxLength",
            StringMaxLength::build,
            StringMaxLength::allowed_types(),
        )?;
        state.add_attribute(
            "pattern",
            StringPattern::build,
            StringPattern::allowed_types(),
        )?;

        // Number & Integer
        state.add_attribute(
            "min",
            NumericMin::build_inclusive,
            NumericMin::allowed_types(),
        )?;
        state.add_attribute(
            "exclusiveMin",
            NumericMin::build_exclusive,
            NumericMin::allowed_types(),
        )?;
        state.add_attribute(
            "max",
            NumericMax::build_inclusive,
            NumericMax::allowed_types(),
        )?;
        state.add_attribute(
            "exclusiveMax",
            NumericMax::build_exclusive,
            NumericMax::allowed_types(),
        )?;

        // Object
        state.add_attribute(
            "properties",
            ObjectProperties::build,
            ObjectProperties::allowed_types(),
        )?;

        // Array
        state.add_attribute("items", ArrayItems::build, ArrayItems::allowed_types())?;

        // Const
        state.add_attribute("value", ConstValue::build, ConstValue::allowed_types())?;

        // Enum
        state.add_attribute("values", EnumValues::build, EnumValues::allowed_types())?;

        // Pointer
        state.add_attribute("path", PointerPath::build, PointerPath::allowed_types())?;

        Ok(state)
    }

    pub fn add_definition(&mut self, def: Definition, path: DocumentPath) -> Result<(), Error> {
        let ptr = def.ptr();
        if let Some(_) = self.defs.get(ptr.as_str()) {
            return Err(Error::DuplicatePointer { path, ptr });
        }
        if self.unresolved_ptrs.get(ptr.as_str()).is_some() {
            self.unresolved_ptrs.remove(ptr.as_str());
        }
        self.defs.insert(ptr, def);
        Ok(())
    }

    pub fn add_attribute(
        &mut self,
        name: &str,
        builder: NewAttribute,
        allowed_types: HashSet<AllowedType>,
    ) -> Result<&Self, Error> {
        for t in allowed_types {
            match self.types.get_mut(&t.typ()) {
                Some(typ) => {
                    typ.insert(name.to_string(), builder);
                }
                None => {
                    let mut map = HashMap::<String, NewAttribute>::new();
                    map.insert(name.to_string(), builder);
                    self.types.insert(t.typ(), map);
                }
            };
            if t.required() {
                if let None = self.required_attrs.get(&t.typ()) {
                    self.required_attrs.insert(t.typ(), HashSet::new());
                }
                if let Some(s) = self.required_attrs.get_mut(&t.typ()) {
                    s.insert(name.to_string());
                }
            }
        }

        Ok(self)
    }

    pub fn add_unresolved_pointer(&mut self, target: String, source: String) {
        if self.defs.get(target.as_str()).is_none() {
            if let None = self.unresolved_ptrs.get(target.as_str()) {
                self.unresolved_ptrs.insert(target.clone(), HashSet::new());
            }
            if let Some(s) = self.unresolved_ptrs.get_mut(target.as_str()) {
                s.insert(source);
            }
        }
    }

    pub fn get_definition(&self, ptr: &str) -> Option<&Definition> {
        match self.defs.get(ptr) {
            Some(d) => Some(d),
            None => None,
        }
    }

    pub fn get_attribute_builder(&self, typ: Type, attr: &str) -> Option<NewAttribute> {
        match self.types.get(&typ) {
            Some(attrs) => match attrs.get(attr) {
                Some(b) => Some(*b),
                None => None,
            },
            None => None,
        }
    }

    pub fn get_required_attributes(&self, typ: Type) -> Option<&HashSet<String>> {
        match self.required_attrs.get(&typ) {
            Some(s) => Some(&s),
            None => None,
        }
    }

    pub fn get_unresolved_pointer(&self) -> Option<(String, HashSet<String>)> {
        for (target, sources) in &self.unresolved_ptrs {
            return Some((target.clone(), sources.clone()));
        }
        None
    }
}

pub struct Context<'j> {
    ptr: String,
    path: Vec<&'j serde_json::Value>, // TODO: Move to DocumentPath.
    raw_definition: &'j serde_json::Map<String, serde_json::Value>,
    name: String,
}

impl<'j> Context<'j> {
    pub fn new(
        ptr: String,
        path: Vec<&'j serde_json::Value>,
        raw_definition: &'j serde_json::Map<String, serde_json::Value>,
        name: &str,
    ) -> Self {
        Context {
            ptr,
            path,
            raw_definition,
            name: name.to_string(),
        }
    }

    pub fn ptr(&self) -> String {
        self.ptr.clone()
    }

    pub fn path(&self) -> Vec<&serde_json::Value> {
        self.path.clone()
    }

    pub fn raw_definition(&self) -> &serde_json::Map<String, serde_json::Value> {
        self.raw_definition
    }

    pub fn name(&self) -> String {
        self.name.clone()
    }
}

pub struct Validator {
    state: State,
}

impl Validator {
    pub fn new() -> Result<Self, Error> {
        Ok(Validator {
            state: State::new()?,
        })
    }

    pub fn load_str(mut self, json: &str) -> Result<Self, Error> {
        let json = match serde_json::from_str(json) {
            Ok(json) => json,
            Err(e) => return Err(Error::ParseError(e)),
        };
        self.build_document(&json)?;
        Ok(self)
    }

    pub fn load_json(mut self, json: &serde_json::Value) -> Result<Self, Error> {
        self.build_document(json)?;
        Ok(self)
    }

    // TODO: pub fn load_file(&mut self) {}

    pub fn finalize(self) -> Result<Self, Error> {
        match self.state.get_unresolved_pointer() {
            Some((target, sources)) => Err(Error::UnresolvedPointers { target, sources }),
            None => Ok(self),
        }
    }

    pub fn validate_json(
        &self,
        ptr: &str,
        json: &serde_json::Value,
    ) -> Result<(), ValidationError> {
        self.validate(ptr, &json)?;
        Ok(())
    }

    // TODO: pub fn validate_str("/pointer/to/definition", json_str);

    fn validate(&self, ptr: &str, input: &serde_json::Value) -> Result<(), ValidationError> {
        let def = match self.state.get_definition(ptr) {
            Some(def) => def,
            None => return Err(ValidationError::UndefinedDefinition),
        };

        def.validate(&self.state, input, vec![])
    }

    fn build_document(&mut self, input: &serde_json::Value) -> Result<(), Error> {
        let doc = match input.as_object() {
            Some(d) => d,
            None => return Err(Error::InvalidRoot),
        };

        let id = match doc.get("id") {
            Some(id) => match id.as_str() {
                Some(id_str) => id_str,
                None => {
                    let mut path = DocumentPath::new("");
                    path.add("id");
                    return Err(Error::InvalidValue {
                        path,
                        value: id.clone(),
                    });
                }
            },
            None => {
                return Err(Error::MissingAttribute {
                    path: DocumentPath::new(""),
                    attr: "id".to_string(),
                })
            }
        };
        let root_ptr = id.to_string();
        let mut path = DocumentPath::new(id);

        let definitions = match doc.get("definitions") {
            Some(d) => match d.as_array() {
                Some(defs) => defs,
                None => {
                    path.add("definitions");
                    return Err(Error::InvalidValue {
                        path,
                        value: d.clone(),
                    });
                }
            },
            None => {
                return Err(Error::MissingAttribute {
                    path,
                    attr: "definitions".to_string(),
                })
            }
        };

        path.add("definitions");

        for (i, d) in definitions.iter().enumerate() {
            let mut path = path.clone();
            path.add(i.to_string().as_str());
            let def = Definition::new(&mut self.state, d, root_ptr.clone(), path.clone(), vec![])?;
            self.state.add_definition(def, path)?;
        }

        Ok(())
    }
}
