use super::{AllowedType, Attribute};
use crate::definition;
use crate::error::{Error, ValidationError};
use crate::validator::{Context, DocumentPath, State};

use std::collections::HashSet;

use serde_json;

#[derive(Debug)]
pub struct DefinitionType {
    name: String,
    typ: definition::Type,
}

impl DefinitionType {
    pub fn new(path: DocumentPath, ctx: &Context) -> Result<Self, Error> {
        let obj = ctx.raw_definition();
        let typ = definition::Type::new(obj, path.clone())?;
        Ok(DefinitionType {
            name: ctx.name(),
            typ,
        })
    }

    pub fn build(
        _: &mut State,
        path: DocumentPath,
        ctx: &Context,
    ) -> Result<Box<Attribute>, Error> {
        Ok(Box::new(DefinitionType::new(path, ctx)?))
    }

    pub fn allowed_types() -> HashSet<AllowedType> {
        use crate::definition::Type::*;
        let mut set: HashSet<AllowedType> = HashSet::new();
        set.insert(AllowedType::new(Null, true));
        set.insert(AllowedType::new(Boolean, true));
        set.insert(AllowedType::new(String, true));
        set.insert(AllowedType::new(Number, true));
        set.insert(AllowedType::new(Integer, true));
        set.insert(AllowedType::new(Object, true));
        set.insert(AllowedType::new(Array, true));
        set.insert(AllowedType::new(Enum, true));
        set.insert(AllowedType::new(Const, true));
        set.insert(AllowedType::new(Pointer, true));
        set
    }
}

impl Attribute for DefinitionType {
    fn validate(
        &self,
        _: &State,
        path: Vec<String>,
        input: &serde_json::Value,
    ) -> Result<(), ValidationError> {
        use crate::definition::Type::*;

        let err_msg = match self.typ {
            Null => {
                if input.is_null() {
                    return Ok(());
                }
                "Value must be null."
            }
            Boolean => {
                if input.is_boolean() {
                    return Ok(());
                }
                "Value must be boolean."
            }
            String => {
                if input.is_string() {
                    return Ok(());
                }
                "Value must be a string."
            }
            Number => {
                if input.is_number() {
                    return Ok(());
                }
                "Value must be a number."
            }
            Integer => {
                if input.is_i64() {
                    return Ok(());
                }
                "Value must be a integer."
            }
            Object => {
                if input.is_object() {
                    return Ok(());
                }
                "Value must be an object."
            }
            Array => {
                if input.is_array() {
                    return Ok(());
                }
                "Value must be an array."
            }
            _ => return Ok(()), // Enum, const and pointer types rely on other attributes.
        };

        Err(ValidationError::Failure {
            rule: self.name.clone(),
            path: path,
            message: err_msg.to_string(),
        })
    }
}
