use super::{AllowedType, Attribute};
use crate::definition::Type;
use crate::error::{Error, ValidationError};
use crate::validator::{Context, DocumentPath, State};

use std::collections::HashSet;

use regex::Regex;

#[derive(Debug)]
pub struct StringPattern {
    name: String,
    re: Regex,
}

impl StringPattern {
    pub fn new(mut path: DocumentPath, ctx: &Context) -> Result<Self, Error> {
        let obj = ctx.raw_definition();

        match Type::new(obj, path.clone())? {
            Type::String => (),
            typ => return Err(Error::ForbiddenType { path, typ }),
        };

        let re = match obj.get(ctx.name().as_str()) {
            Some(pattern) => match pattern.as_str() {
                Some(pattern_str) => match Regex::new(pattern_str) {
                    Ok(re) => re,
                    Err(_) => {
                        return Err(Error::InvalidValue {
                            path,
                            value: pattern.clone(),
                        })
                    }
                },
                None => {
                    path.add(ctx.name().as_str());
                    return Err(Error::InvalidValue {
                        path,
                        value: pattern.clone(),
                    });
                }
            },
            None => {
                return Err(Error::MissingAttribute {
                    path,
                    attr: ctx.name(),
                })
            }
        };

        Ok(StringPattern {
            name: ctx.name(),
            re,
        })
    }

    pub fn allowed_types() -> HashSet<AllowedType> {
        let mut set = HashSet::<AllowedType>::new();
        set.insert(AllowedType::new(Type::String, false));
        set
    }

    pub fn build(
        _: &mut State,
        path: DocumentPath,
        ctx: &Context,
    ) -> Result<Box<Attribute>, Error> {
        Ok(Box::new(StringPattern::new(path, ctx)?))
    }
}

impl Attribute for StringPattern {
    fn validate(
        &self,
        _: &State,
        path: Vec<String>,
        input: &serde_json::Value,
    ) -> Result<(), ValidationError> {
        let val = match input.as_str() {
            Some(val) => val,
            None => {
                return Err(ValidationError::Failure {
                    rule: "type".to_string(),
                    path: path,
                    message: "Value must be a string.".to_string(),
                })
            }
        };
        if !self.re.is_match(val) {
            return Err(ValidationError::Failure {
                rule: self.name.clone(),
                path: path,
                message: "Value does not match the required format.".to_string(),
            });
        }
        Ok(())
    }
}
