use super::{AllowedType, Attribute};
use crate::definition::Type;
use crate::error::{Error, ValidationError};
use crate::validator::{Context, DocumentPath, State};

use std::collections::HashSet;

use serde_json;

#[derive(Debug)]
pub struct EnumValues {
    name: String,
    values: Vec<serde_json::Value>,
}

impl EnumValues {
    pub fn new(mut path: DocumentPath, ctx: &Context) -> Result<Self, Error> {
        let obj = ctx.raw_definition();

        match Type::new(obj, path.clone())? {
            Type::Enum => (),
            typ => return Err(Error::ForbiddenType { path, typ }),
        };

        let values = match obj.get(ctx.name().as_str()) {
            Some(values) => match values.as_array() {
                Some(values_arr) => values_arr,
                None => {
                    path.add(ctx.name().as_str());
                    return Err(Error::InvalidValue {
                        path,
                        value: values.clone(),
                    });
                }
            },
            None => {
                return Err(Error::MissingAttribute {
                    path,
                    attr: ctx.name(),
                })
            }
        };

        Ok(EnumValues {
            name: ctx.name(),
            values: values.clone(),
        })
    }

    pub fn allowed_types() -> HashSet<AllowedType> {
        let mut set = HashSet::<AllowedType>::new();
        set.insert(AllowedType::new(Type::Enum, true));
        set
    }

    pub fn build(
        _: &mut State,
        path: DocumentPath,
        ctx: &Context,
    ) -> Result<Box<Attribute>, Error> {
        Ok(Box::new(EnumValues::new(path, ctx)?))
    }
}

impl Attribute for EnumValues {
    fn validate(
        &self,
        _: &State,
        path: Vec<String>,
        input: &serde_json::Value,
    ) -> Result<(), ValidationError> {
        for v in &self.values {
            if v == input {
                return Ok(());
            }
        }
        return Err(ValidationError::Failure {
            rule: "value".to_string(),
            path: path,
            message: "Value is invalid.".to_string(),
        });
    }
}
