use super::{AllowedType, Attribute};
use crate::definition::Type;
use crate::error::{Error, ValidationError};
use crate::validator::{Context, DocumentPath, State};

use std::collections::HashSet;

use serde_json;

#[derive(Debug)]
pub struct ConstValue {
    name: String,
    value: serde_json::Value,
}

impl ConstValue {
    pub fn new(path: DocumentPath, ctx: &Context) -> Result<Self, Error> {
        let obj = ctx.raw_definition();

        match Type::new(obj, path.clone())? {
            Type::Const => (),
            typ => return Err(Error::ForbiddenType { path, typ }),
        };

        let value = match obj.get(ctx.name().as_str()) {
            Some(value) => value.clone(),
            None => {
                return Err(Error::MissingAttribute {
                    path,
                    attr: ctx.name(),
                })
            }
        };

        Ok(ConstValue {
            name: ctx.name(),
            value,
        })
    }

    pub fn allowed_types() -> HashSet<AllowedType> {
        let mut set = HashSet::<AllowedType>::new();
        set.insert(AllowedType::new(Type::Const, true));
        set
    }

    pub fn build(
        _: &mut State,
        path: DocumentPath,
        ctx: &Context,
    ) -> Result<Box<Attribute>, Error> {
        Ok(Box::new(ConstValue::new(path, ctx)?))
    }
}

impl Attribute for ConstValue {
    fn validate(
        &self,
        _: &State,
        path: Vec<String>,
        input: &serde_json::Value,
    ) -> Result<(), ValidationError> {
        if self.value != *input {
            return Err(ValidationError::Failure {
                rule: "value".to_string(),
                path: path,
                message: format!("Value must be {:?}.", self.value),
            });
        }
        Ok(())
    }
}
