mod array_items;
mod const_value;
mod enum_values;
mod numeric_max;
mod numeric_min;
mod object_properties;
mod pointer_path;
mod string_max_length;
mod string_min_length;
mod string_pattern;
mod typ;

pub use self::array_items::*;
pub use self::const_value::*;
pub use self::enum_values::*;
pub use self::numeric_max::*;
pub use self::numeric_min::*;
pub use self::object_properties::*;
pub use self::pointer_path::*;
pub use self::string_max_length::*;
pub use self::string_min_length::*;
pub use self::string_pattern::*;
pub use self::typ::*;

use crate::definition;
use crate::error::{Error, ValidationError};
use crate::validator::{DocumentPath, State};

use std::fmt::Debug;
use std::hash::{Hash, Hasher};

use serde_json;

pub trait Attribute: Debug {
    fn validate(
        &self,
        state: &State,
        path: Vec<String>, // Path of the input JSON
        input: &serde_json::Value,
    ) -> Result<(), ValidationError>;
}

pub struct AllowedType {
    typ: definition::Type,
    required: bool,
}

impl AllowedType {
    pub fn new(typ: definition::Type, required: bool) -> Self {
        AllowedType { typ, required }
    }

    pub fn typ(&self) -> definition::Type {
        self.typ.clone()
    }

    pub fn required(&self) -> bool {
        self.required
    }
}

impl PartialEq for AllowedType {
    fn eq(&self, other: &AllowedType) -> bool {
        self.typ == other.typ
    }
}
impl Eq for AllowedType {}

impl Hash for AllowedType {
    fn hash<H: Hasher>(&self, state: &mut H) {
        self.typ.hash(state);
    }
}

pub fn extract_id(
    obj: &serde_json::Map<String, serde_json::Value>,
    path: &mut DocumentPath,
) -> Result<String, Error> {
    match obj.get("id") {
        Some(id) => match id.as_str() {
            Some(id_str) => Ok(id_str.to_string()),
            None => {
                path.add("id");
                return Err(Error::InvalidValue {
                    path: path.clone(),
                    value: id.clone(),
                });
            }
        },
        None => {
            return Err(Error::MissingAttribute {
                path: path.clone(),
                attr: "id".to_string(),
            })
        }
    }
}
