#![recursion_limit = "128"]

extern crate json_model;
#[macro_use]
extern crate serde_json;

use json_model::*;

#[test]
fn validate_json() {
    let doc = json!({
        "id": "model",
        "definitions": [
            {
                "id": "noop",
                "type": "null",
            },
            {
                "id": "binary",
                "type": "boolean",
            },
            {
                "id": "hex_color",
                "type": "string",
                "minLength": 4,
                "maxLength": 7,
                "pattern": "^#(?:[0-9a-fA-F]{3}){1,2}$",
            },
            {
                "id": "temperature",
                "type": "number",
                "min": -1.23,
                "max": 76,
            },
            {
                "id": "age",
                "type": "integer",
                "min": 0,
                "max": 150,
            },
            {
                "id": "price",
                "type": "number",
                "exclusiveMin": 0,
                "exclusiveMax": 10,
            },
            {
                "id": "height",
                "type": "integer",
                "exclusiveMin": 0,
                "exclusiveMax": 273,
            },
            {
                "id": "user",
                "type": "object",
                // TODO: "nestingLimit": 4,
                // TODO: "maxProperties": 10,
                // TODO: "minProperties": 3,
                "properties": { // Attributes can have arbitrary sub-attributes. 
                    "strict": true,
                    "names": {
                        "username": "uname",
                        "password": "pwd",
                    },
                    "namePatterns": {
                        "first_name": "^(fname|1st_name)$",
                    },
                    "required": ["username", "password"],
                    "definitions": [
                        {
                            "id": "username",
                            "type": "string",
                            "minLength": 1,
                            "maxLength": 30,
                        },
                        {
                            "id": "password",
                            "type": "string",
                            "minLength": 5,
                            "maxLength": 256,
                        },
                        {
                            "id": "first_name",
                            "type": "string",
                        },
                        {
                            "id": "last_name",
                            "type": "string",
                        },
                    ]
                }
            },
            {
                "id": "array",
                "type": "array",
                // TODO: "unique": true,
                // TODO: "minLength": 1,
                // TODO: "maxLength": 4,
                "items": [
                    {
                        "id": "string",
                        "type": "string",
                    },
                    {
                        "id": "number",
                        "type": "number",
                    },
                    {
                        "id": "boolean",
                        "type": "boolean",
                    }
                ]
            },
            {
                "id": "enum",
                "type": "enum",
                "values": [1, { "2": 3 }, "4"],
            },
            {
                "id": "const",
                "type": "const",
                "value": [1, { "2": 3 }, "4"],
            },
            {
                "id": "username_ptr",
                "type": "pointer",
                "path": "model/user/username",
            }
        ]
    });

    let v = Validator::new()
        .unwrap()
        .load_json(&doc)
        .unwrap()
        .finalize()
        .unwrap();

    // Null
    assert!(v.validate_json("model/noop", &json!(())).is_ok());
    assert!(v.validate_json("model/noop", &json!(1)).is_err());

    // Boolean
    assert!(v.validate_json("model/binary", &json!(true)).is_ok());
    assert!(v.validate_json("model/binary", &json!(false)).is_ok());
    assert!(v.validate_json("model/binary", &json!(1)).is_err());

    // String
    assert!(v.validate_json("model/hex_color", &json!("123")).is_err());
    assert!(v.validate_json("model/hex_color", &json!("#xyz")).is_err());
    assert!(v.validate_json("model/hex_color", &json!("#123")).is_ok());
    assert!(v
        .validate_json("model/hex_color", &json!("#123456"))
        .is_ok());
    assert!(v
        .validate_json("model/hex_color", &json!("#1234567"))
        .is_err());
    assert!(v.validate_json("model/hex_color", &json!(1)).is_err());

    // Number
    assert!(v.validate_json("model/temperature", &json!(-1.23)).is_ok());
    assert!(v.validate_json("model/temperature", &json!(76)).is_ok());
    assert!(v.validate_json("model/temperature", &json!(-2)).is_err());
    assert!(v.validate_json("model/temperature", &json!(false)).is_err());
    assert!(v.validate_json("model/temperature", &json!(76.5)).is_err());

    // Number exclusive
    assert!(v.validate_json("model/price", &json!(0.05)).is_ok());
    assert!(v.validate_json("model/price", &json!(0)).is_err());
    assert!(v.validate_json("model/price", &json!(10)).is_err());

    // Integer
    assert!(v.validate_json("model/age", &json!(0)).is_ok());
    assert!(v.validate_json("model/age", &json!(150)).is_ok());
    assert!(v.validate_json("model/age", &json!(-1)).is_err());
    assert!(v.validate_json("model/age", &json!(151)).is_err());

    // Integer exclusive
    assert!(v.validate_json("model/height", &json!(1)).is_ok());
    assert!(v.validate_json("model/height", &json!(272)).is_ok());
    assert!(v.validate_json("model/height", &json!(0)).is_err());
    assert!(v.validate_json("model/height", &json!(273)).is_err());

    // Object
    assert!(v
        .validate_json(
            "model/user",
            &json!({
                "fname": "first_name",
                "last_name": "last_name",
                "uname": "username",
                "pwd": "Tr0ub4dor&3",
            })
        )
        .is_ok());
    assert!(v
        .validate_json(
            "model/user",
            &json!({
                "random_property": "random",
                "uname": "username",
                "pwd": "Tr0ub4dor&3",
            })
        )
        .is_err());
    assert!(v
        .validate_json(
            "model/user",
            &json!({
                "pwd": "Tr0ub4dor&3",
            })
        )
        .is_err());

    // Array
    assert!(v
        .validate_json("model/array", &json!(["1", 2, false]))
        .is_ok());
    assert!(v.validate_json("model/array", &json!([()])).is_err());

    // Enum
    assert!(v.validate_json("model/enum", &json!(1)).is_ok());
    assert!(v.validate_json("model/enum", &json!({ "2": 3 })).is_ok());
    assert!(v.validate_json("model/enum", &json!("4")).is_ok());
    assert!(v.validate_json("model/enum", &json!(0)).is_err());

    // Const
    assert!(v
        .validate_json("model/const", &json!([1, { "2": 3 }, "4"]))
        .is_ok());
    assert!(v.validate_json("model/const", &json!(1234)).is_err());

    // Pointer
    assert!(v.validate_json("model/username_ptr", &json!("ptr")).is_ok());
    assert!(v.validate_json("model/username_ptr", &json!("")).is_err());
    assert!(v.validate_json("model/username_ptr", &json!(0)).is_err());
}
