use crate::attribute::{extract_id, Attribute};
use crate::error::{Error, ValidationError};
use crate::validator::{Context, DocumentPath, State};

use serde_json;

#[derive(Clone, Debug, Eq, Hash, PartialEq)]
pub enum Type {
    Null,
    Boolean,
    String,
    Number,
    Integer,
    Object,
    Array,
    Enum,
    Const,
    Pointer,
}

impl Type {
    pub fn new(
        obj: &serde_json::Map<String, serde_json::Value>,
        mut path: DocumentPath,
    ) -> Result<Self, Error> {
        let t = match obj.get("type") {
            Some(t) => match t.as_str() {
                Some(t) => match t {
                    "null" => Type::Null,
                    "boolean" => Type::Boolean,
                    "string" => Type::String,
                    "number" => Type::Number,
                    "integer" => Type::Integer,
                    "object" => Type::Object,
                    "array" => Type::Array,
                    "enum" => Type::Enum,
                    "const" => Type::Const,
                    "pointer" => Type::Pointer,
                    typ => {
                        path.add("type");
                        return Err(Error::UnknownType {
                            path,
                            typ: typ.to_string(),
                        });
                    }
                },
                None => {
                    path.add("type");
                    return Err(Error::InvalidValue {
                        path,
                        value: t.clone(),
                    });
                }
            },
            None => {
                return Err(Error::MissingAttribute {
                    path,
                    attr: "type".to_string(),
                })
            }
        };
        Ok(t)
    }
}

#[derive(Debug)]
pub struct Definition {
    ptr: String,
    attrs: Vec<Box<Attribute>>,
}

impl Definition {
    pub fn new<'j>(
        state: &mut State,
        input: &'j serde_json::Value,
        mut ptr: String,
        mut path: DocumentPath,
        mut path2: Vec<&'j serde_json::Value>, // TODO: Move it to DocumentPath or elsewhere.
    ) -> Result<Self, Error> {
        let obj = match input.as_object() {
            Some(o) => o,
            None => {
                return Err(Error::InvalidValue {
                    path,
                    value: input.clone(),
                })
            }
        };

        let typ = Type::new(obj, path.clone())?;
        let id = extract_id(obj, &mut path)?;
        ptr.push_str("/");
        ptr.push_str(id.as_str());

        let mut attrs: Vec<Box<Attribute>> = vec![];

        match state.get_required_attributes(typ.clone()) {
            Some(set) => {
                for a in set {
                    if let None = obj.get(a.as_str()) {
                        return Err(Error::MissingAttribute {
                            path,
                            attr: a.to_string(),
                        });
                    }
                }
            }
            None => (),
        }

        path2.push(input);

        for (k, _) in obj {
            match k.as_str() {
                "id" => continue,
                attr => {
                    let build = match state.get_attribute_builder(typ.clone(), attr) {
                        Some(b) => b,
                        None => {
                            return Err(Error::UnrecognizedAttribute {
                                path,
                                attr: attr.to_string(),
                            })
                        }
                    };
                    let ctx = Context::new(ptr.clone(), path2.clone(), obj, attr);
                    let attr = build(state, path.clone(), &ctx)?;
                    attrs.push(attr);
                }
            }
        }

        Ok(Definition { ptr, attrs })
    }

    pub fn validate(
        &self,
        state: &State,
        input: &serde_json::Value,
        path: Vec<String>,
    ) -> Result<(), ValidationError> {
        for a in &self.attrs {
            a.validate(state, path.clone(), input)?;
        }
        Ok(())
    }

    pub fn ptr(&self) -> String {
        self.ptr.clone()
    }
}
