use super::{AllowedType, Attribute};
use crate::definition::Type;
use crate::error::{Error, ValidationError};
use crate::validator::{Context, DocumentPath, State};

use std::collections::HashSet;

#[derive(Debug)]
pub struct StringMinLength {
    name: String,
    value: usize,
}

impl StringMinLength {
    pub fn new(mut path: DocumentPath, ctx: &Context) -> Result<Self, Error> {
        let obj = ctx.raw_definition();

        match Type::new(obj, path.clone())? {
            Type::String => (),
            typ => return Err(Error::ForbiddenType { path, typ }),
        };

        let value = match obj.get(ctx.name().as_str()) {
            Some(len) => match len.as_u64() {
                Some(len) => len,
                None => {
                    path.add(ctx.name().as_str());
                    return Err(Error::InvalidValue {
                        path,
                        value: len.clone(),
                    });
                }
            },
            None => {
                return Err(Error::MissingAttribute {
                    path,
                    attr: ctx.name(),
                })
            }
        };

        Ok(StringMinLength {
            name: ctx.name(),
            value: value as usize,
        })
    }

    pub fn allowed_types() -> HashSet<AllowedType> {
        let mut set = HashSet::<AllowedType>::new();
        set.insert(AllowedType::new(Type::String, false));
        set
    }

    pub fn build(
        _: &mut State,
        path: DocumentPath,
        ctx: &Context,
    ) -> Result<Box<Attribute>, Error> {
        Ok(Box::new(StringMinLength::new(path, ctx)?))
    }
}

impl Attribute for StringMinLength {
    fn validate(
        &self,
        _: &State,
        path: Vec<String>,
        input: &serde_json::Value,
    ) -> Result<(), ValidationError> {
        let val = match input.as_str() {
            Some(val) => val,
            None => {
                return Err(ValidationError::Failure {
                    rule: "type".to_string(),
                    path: path,
                    message: "Value must be a string.".to_string(),
                })
            }
        };
        if val.len() < self.value {
            return Err(ValidationError::Failure {
                rule: self.name.clone(),
                path: path,
                message: format!("Value must be at least {} characters long.", self.value),
            });
        }
        Ok(())
    }
}
