use super::{AllowedType, Attribute};
use crate::definition::Type;
use crate::error::{Error, ValidationError};
use crate::validator::{Context, DocumentPath, State};

use std::collections::HashSet;

#[derive(Debug)]
pub struct PointerPath {
    name: String,
    ptr: String,
}

impl PointerPath {
    pub fn new(state: &mut State, mut path: DocumentPath, ctx: &Context) -> Result<Self, Error> {
        let obj = ctx.raw_definition();

        match Type::new(obj, path.clone())? {
            Type::Pointer => (),
            typ => return Err(Error::ForbiddenType { path, typ }),
        };

        let ptr = match obj.get(ctx.name().as_str()) {
            Some(ptr_path) => match ptr_path.as_str() {
                Some(path) => path,
                None => {
                    path.add(ctx.name().as_str());
                    return Err(Error::InvalidValue {
                        path,
                        value: ptr_path.clone(),
                    });
                }
            },
            None => {
                return Err(Error::MissingAttribute {
                    path,
                    attr: ctx.name(),
                })
            }
        };

        state.add_unresolved_pointer(ptr.to_string(), ctx.ptr());

        Ok(PointerPath {
            name: ctx.name(),
            ptr: ptr.to_string(),
        })
    }

    pub fn allowed_types() -> HashSet<AllowedType> {
        let mut set = HashSet::<AllowedType>::new();
        set.insert(AllowedType::new(Type::Pointer, true));
        set
    }

    pub fn build(
        state: &mut State,
        path: DocumentPath,
        ctx: &Context,
    ) -> Result<Box<Attribute>, Error> {
        Ok(Box::new(PointerPath::new(state, path, ctx)?))
    }
}

impl Attribute for PointerPath {
    fn validate(
        &self,
        state: &State,
        path: Vec<String>,
        input: &serde_json::Value,
    ) -> Result<(), ValidationError> {
        let def = match state.get_definition(self.ptr.as_str()) {
            Some(def) => def,
            None => return Err(ValidationError::UndefinedDefinition),
        };

        def.validate(state, input, path)?;

        Ok(())
    }
}
