use super::{extract_id, AllowedType, Attribute};
use crate::definition::{Definition, Type};
use crate::error::{Error, ValidationError};
use crate::validator::{Context, DocumentPath, State};

use std::collections::{HashMap, HashSet};

use regex::Regex;
use serde_json;

#[derive(Debug)]
struct NamePattern {
    re: Regex,
    ptr: String,
}

const REQUIRED_RULE: &str = "required";
const STRICT_RULE: &str = "strict";
const NAME_RULE: &str = "name";

#[derive(Debug)]
pub struct ObjectProperties {
    name: String,
    strict: bool,
    names: HashMap<String, String>, // name/ptr
    name_patterns: Vec<NamePattern>,
    required_names: HashSet<String>,
    required_name_patterns: Vec<Regex>,
}

impl ObjectProperties {
    pub fn new(state: &mut State, mut path: DocumentPath, ctx: &Context) -> Result<Self, Error> {
        let obj = ctx.raw_definition();

        match Type::new(obj, path.clone())? {
            Type::Object => (),
            typ => return Err(Error::ForbiddenType { path, typ }),
        };

        let props = match obj.get(ctx.name().as_str()) {
            Some(props) => match props.as_object() {
                Some(props_obj) => props_obj,
                None => {
                    path.add(ctx.name().as_str());
                    return Err(Error::InvalidValue {
                        path,
                        value: props.clone(),
                    });
                }
            },
            None => {
                return Err(Error::MissingAttribute {
                    path,
                    attr: ctx.name(),
                })
            }
        };

        path.add(ctx.name().as_str());

        let strict = match props.get("strict") {
            Some(strict) => match strict.as_bool() {
                Some(s) => s,
                None => {
                    path.add("strict");
                    return Err(Error::InvalidValue {
                        path,
                        value: strict.clone(),
                    });
                }
            },
            None => true, // default
        };

        let defs = match props.get("definitions") {
            Some(defs) => match defs.as_array() {
                Some(d) => d,
                None => {
                    path.add("definitions");
                    return Err(Error::InvalidValue {
                        path,
                        value: defs.clone(),
                    });
                }
            },
            None => {
                return Err(Error::MissingAttribute {
                    path,
                    attr: "definitions".to_string(),
                })
            }
        };

        let mut ids = HashMap::<String, String>::new();
        let mut names = HashMap::<String, String>::new();
        let mut name_patterns = Vec::<NamePattern>::new();
        let mut required_names = HashSet::<String>::new();
        let mut required_name_patterns = Vec::<Regex>::new();

        for (i, d) in defs.iter().enumerate() {
            let mut path = path.clone();
            path.add(i.to_string().as_str());
            let obj = match d.as_object() {
                Some(o) => o,
                None => {
                    return Err(Error::InvalidValue {
                        path,
                        value: d.clone(),
                    });
                }
            };
            let def = Definition::new(state, d, ctx.ptr(), path.clone(), ctx.path())?;
            let id = extract_id(obj, &mut path)?;
            if ids.contains_key(&id) {
                path.add("id");
                return Err(Error::DuplicateId { path, id });
            }
            ids.insert(id.clone(), def.ptr());
            names.insert(id.clone(), def.ptr());
            state.add_definition(def, path)?;
        }

        match props.get("required") {
            Some(req) => match req.as_array() {
                Some(req) => {
                    let mut path = path.clone();
                    path.add("required");
                    for (i, id) in req.iter().enumerate() {
                        let mut path = path.clone();
                        path.add(i.to_string().as_str());
                        let id = match id.as_str() {
                            Some(id_str) => id_str,
                            None => {
                                return Err(Error::InvalidValue {
                                    path,
                                    value: id.clone(),
                                })
                            }
                        };
                        if !ids.contains_key(id) {
                            return Err(Error::UndefinedId {
                                path,
                                id: id.to_string(),
                            });
                        }
                        required_names.insert(id.to_string());
                    }
                }
                None => {
                    path.add("required");
                    return Err(Error::InvalidValue {
                        path,
                        value: req.clone(),
                    });
                }
            },
            None => (),
        };

        match props.get("names") {
            Some(n) => match n.as_object() {
                Some(n) => {
                    let mut path = path.clone();
                    path.add("names");
                    for (id, name) in n {
                        let mut path = path.clone();
                        path.add(id);
                        let ptr = match ids.get(id) {
                            Some(ptr) => ptr,
                            None => {
                                return Err(Error::UndefinedId {
                                    path,
                                    id: id.to_string(),
                                })
                            }
                        };
                        let name = match name.as_str() {
                            Some(name) => name,
                            None => {
                                return Err(Error::InvalidValue {
                                    path,
                                    value: name.clone(),
                                })
                            }
                        };
                        if names.contains_key(name) {
                            return Err(Error::DuplicateName {
                                path,
                                name: name.to_string(),
                            });
                        }
                        names.remove(id);
                        names.insert(name.to_string(), ptr.clone());
                        if required_names.contains(id) {
                            required_names.remove(id);
                            required_names.insert(name.to_string());
                        }
                    }
                }
                None => {
                    path.add("names");
                    return Err(Error::InvalidValue {
                        path,
                        value: n.clone(),
                    });
                }
            },
            None => (),
        };

        match props.get("namePatterns") {
            Some(p) => match p.as_object() {
                Some(p) => {
                    let mut path = path.clone();
                    path.add("namePatterns");
                    for (id, pattern) in p {
                        let mut path = path.clone();
                        path.add(id);
                        let ptr = match ids.get(id) {
                            Some(ptr) => ptr,
                            None => {
                                return Err(Error::UndefinedId {
                                    path,
                                    id: id.to_string(),
                                })
                            }
                        };
                        let pattern_str = match pattern.as_str() {
                            Some(pattern_str) => pattern_str,
                            None => {
                                return Err(Error::InvalidValue {
                                    path,
                                    value: pattern.clone(),
                                })
                            }
                        };
                        let re = match Regex::new(pattern_str) {
                            Ok(re) => re,
                            Err(_) => {
                                return Err(Error::InvalidValue {
                                    path,
                                    value: pattern.clone(),
                                })
                            }
                        };
                        for p in &name_patterns {
                            if p.re.as_str() == re.as_str() {
                                return Err(Error::DuplicateName {
                                    path,
                                    name: pattern_str.to_string(),
                                });
                            }
                        }
                        names.remove(id);
                        name_patterns.push(NamePattern {
                            re: re.clone(),
                            ptr: ptr.to_string(),
                        });
                        if required_names.contains(id) {
                            required_names.remove(id);
                            required_name_patterns.push(re);
                        }
                    }
                }
                None => {
                    path.add("namePatterns");
                    return Err(Error::InvalidValue {
                        path,
                        value: p.clone(),
                    });
                }
            },
            None => (),
        };

        Ok(ObjectProperties {
            name: ctx.name(),
            strict,
            names,
            name_patterns,
            required_names,
            required_name_patterns,
        })
    }

    pub fn allowed_types() -> HashSet<AllowedType> {
        let mut set = HashSet::<AllowedType>::new();
        set.insert(AllowedType::new(Type::Object, true));
        set
    }

    pub fn build(
        state: &mut State,
        path: DocumentPath,
        ctx: &Context,
    ) -> Result<Box<Attribute>, Error> {
        Ok(Box::new(ObjectProperties::new(state, path, ctx)?))
    }

    fn check_strict(
        &self,
        obj: &serde_json::Map<String, serde_json::Value>,
        path: &Vec<String>,
    ) -> Result<(), ValidationError> {
        if self.strict {
            for name in obj.keys() {
                if self.names.get(name).is_none() {
                    let mut found = false;
                    for p in &self.name_patterns {
                        if p.re.is_match(name) {
                            found = true;
                            break;
                        }
                    }
                    if !found {
                        return Err(ValidationError::Failure {
                            rule: STRICT_RULE.to_string(),
                            path: path.clone(),
                            message: format!("Unrecognized property '{}'.", name),
                        });
                    }
                }
            }
        }
        Ok(())
    }

    fn check_required(
        &self,
        obj: &serde_json::Map<String, serde_json::Value>,
        path: &Vec<String>,
    ) -> Result<(), ValidationError> {
        for name in &self.required_names {
            if let None = obj.get(name.as_str()) {
                return Err(ValidationError::Failure {
                    rule: REQUIRED_RULE.to_string(),
                    path: path.clone(),
                    message: format!("{} is required.", name),
                });
            }
        }

        for re in &self.required_name_patterns {
            let mut found = false;
            for name in obj.keys() {
                if re.is_match(name) {
                    found = true;
                    break;
                }
            }
            if !found {
                return Err(ValidationError::Failure {
                    rule: REQUIRED_RULE.to_string(),
                    path: path.clone(),
                    message: format!("A property is required to match {}.", re.as_str()),
                });
            }
        }
        Ok(())
    }
}

impl Attribute for ObjectProperties {
    fn validate(
        &self,
        state: &State,
        path: Vec<String>,
        input: &serde_json::Value,
    ) -> Result<(), ValidationError> {
        let obj = match input.as_object() {
            Some(obj) => obj,
            None => {
                return Err(ValidationError::Failure {
                    rule: "type".to_string(),
                    path: path,
                    message: "Value must be an object.".to_string(),
                })
            }
        };

        self.check_strict(obj, &path)?;
        self.check_required(obj, &path)?;
        for (name, value) in obj {
            let ptr = match self.names.get(name) {
                Some(ptr) => ptr,
                None => {
                    let mut ptr: Option<&String> = None;
                    for p in &self.name_patterns {
                        if p.re.is_match(name) {
                            ptr = Some(&p.ptr);
                        }
                    }
                    match ptr {
                        Some(ptr) => ptr,
                        None => {
                            return Err(ValidationError::Failure {
                                rule: NAME_RULE.to_string(),
                                path: path,
                                message: format!("Unrecognized name '{}'.", name),
                            })
                        }
                    }
                }
            };
            let def = match state.get_definition(ptr) {
                Some(def) => def,
                None => return Err(ValidationError::UndefinedDefinition),
            };
            let mut path = path.clone();
            path.push(name.to_string());
            def.validate(state, value, path)?;
        }

        Ok(())
    }
}
