use super::{AllowedType, Attribute};
use crate::definition::Type;
use crate::error::{Error, ValidationError};
use crate::validator::{Context, DocumentPath, State};

use std::collections::HashSet;

use serde_json;

#[derive(Clone, Debug, PartialEq)]
pub enum NumericType {
    Number,
    Integer,
}

#[derive(Debug)]
pub struct NumericMax {
    name: String,
    value: f64,
    typ: NumericType,
    is_exclusive: bool,
}

impl NumericMax {
    pub fn new(is_exclusive: bool, mut path: DocumentPath, ctx: &Context) -> Result<Self, Error> {
        let obj = ctx.raw_definition();

        let typ = match Type::new(obj, path.clone())? {
            Type::Number => NumericType::Number,
            Type::Integer => NumericType::Integer,
            typ => return Err(Error::ForbiddenType { path, typ }),
        };

        let value = match obj.get(ctx.name().as_str()) {
            Some(value) => {
                path.add(ctx.name().as_str());
                extract_numeric(value, typ.clone(), path)?
            }
            None => {
                return Err(Error::MissingAttribute {
                    path,
                    attr: ctx.name(),
                })
            }
        };

        Ok(NumericMax {
            name: ctx.name(),
            value,
            typ,
            is_exclusive,
        })
    }

    pub fn allowed_types() -> HashSet<AllowedType> {
        let mut set = HashSet::<AllowedType>::new();
        set.insert(AllowedType::new(Type::Number, false));
        set.insert(AllowedType::new(Type::Integer, false));
        set
    }

    pub fn build_inclusive(
        _: &mut State,
        path: DocumentPath,
        ctx: &Context,
    ) -> Result<Box<Attribute>, Error> {
        Ok(Box::new(NumericMax::new(false, path, ctx)?))
    }

    pub fn build_exclusive(
        _: &mut State,
        path: DocumentPath,
        ctx: &Context,
    ) -> Result<Box<Attribute>, Error> {
        Ok(Box::new(NumericMax::new(true, path, ctx)?))
    }
}

impl Attribute for NumericMax {
    fn validate(
        &self,
        _: &State,
        path: Vec<String>,
        input: &serde_json::Value,
    ) -> Result<(), ValidationError> {
        if self.typ == NumericType::Number {
            let val = match input.as_f64() {
                Some(val) => val,
                None => {
                    return Err(ValidationError::Failure {
                        rule: "type".to_string(),
                        path: path,
                        message: "Value must be a number.".to_string(),
                    })
                }
            };

            if val >= self.value && self.is_exclusive {
                return Err(ValidationError::Failure {
                    rule: self.name.clone(),
                    path: path,
                    message: format!("Value must be less than {}.", self.value),
                });
            } else if val > self.value {
                return Err(ValidationError::Failure {
                    rule: self.name.clone(),
                    path: path,
                    message: format!("Value must be less than or equal to {}.", self.value),
                });
            }
        } else {
            let val = match input.as_i64() {
                Some(val) => val,
                None => {
                    return Err(ValidationError::Failure {
                        rule: "type".to_string(),
                        path: path,
                        message: "Value must be an integer.".to_string(),
                    })
                }
            };

            if val >= self.value as i64 && self.is_exclusive {
                return Err(ValidationError::Failure {
                    rule: self.name.clone(),
                    path: path,
                    message: format!("Value must be less than {}.", self.value as i64),
                });
            } else if val > self.value as i64 {
                return Err(ValidationError::Failure {
                    rule: self.name.clone(),
                    path: path,
                    message: format!("Value must be less than or equal to {}.", self.value as i64),
                });
            }
        }

        Ok(())
    }
}

fn extract_numeric(
    value: &serde_json::Value,
    typ: NumericType,
    path: DocumentPath,
) -> Result<f64, Error> {
    if typ == NumericType::Number {
        match value.as_f64() {
            Some(val) => Ok(val),
            None => Err(Error::InvalidValue {
                path,
                value: value.clone(),
            }),
        }
    } else {
        match value.as_i64() {
            Some(val) => Ok(val as f64),
            None => Err(Error::InvalidValue {
                path,
                value: value.clone(),
            }),
        }
    }
}
