use super::{AllowedType, Attribute};
use crate::definition::{Definition, Type};
use crate::error::{Error, ValidationError};
use crate::validator::{Context, DocumentPath, State};

use std::collections::HashSet;

use serde_json;

#[derive(Debug)]
pub struct ArrayItems {
    name: String,
    ptrs: HashSet<String>,
}

impl ArrayItems {
    pub fn new(state: &mut State, mut path: DocumentPath, ctx: &Context) -> Result<Self, Error> {
        let obj = ctx.raw_definition();

        match Type::new(obj, path.clone())? {
            Type::Array => (),
            typ => return Err(Error::ForbiddenType { path, typ }),
        };

        let defs = match obj.get(ctx.name().as_str()) {
            Some(items) => match items.as_array() {
                Some(items_arr) => items_arr,
                None => {
                    path.add(ctx.name().as_str());
                    return Err(Error::InvalidValue {
                        path,
                        value: items.clone(),
                    });
                }
            },
            None => {
                return Err(Error::MissingAttribute {
                    path,
                    attr: ctx.name(),
                })
            }
        };

        path.add(ctx.name().as_str());

        let mut ptrs = HashSet::<String>::new();
        for (i, d) in defs.iter().enumerate() {
            let mut path = path.clone();
            path.add(i.to_string().as_str());
            let def = Definition::new(state, d, ctx.ptr(), path.clone(), ctx.path())?;
            ptrs.insert(def.ptr());
            state.add_definition(def, path)?;
        }

        Ok(ArrayItems {
            name: ctx.name(),
            ptrs,
        })
    }

    pub fn allowed_types() -> HashSet<AllowedType> {
        let mut set = HashSet::<AllowedType>::new();
        set.insert(AllowedType::new(Type::Array, true));
        set
    }

    pub fn build(
        state: &mut State,
        path: DocumentPath,
        ctx: &Context,
    ) -> Result<Box<Attribute>, Error> {
        Ok(Box::new(ArrayItems::new(state, path, ctx)?))
    }
}

impl Attribute for ArrayItems {
    fn validate(
        &self,
        state: &State,
        path: Vec<String>,
        input: &serde_json::Value,
    ) -> Result<(), ValidationError> {
        let arr = match input.as_array() {
            Some(arr) => arr,
            None => {
                return Err(ValidationError::Failure {
                    rule: "type".to_string(),
                    path: path,
                    message: "Value must be an array.".to_string(),
                })
            }
        };

        for (i, value) in arr.iter().enumerate() {
            let mut found = false;
            for ptr in &self.ptrs {
                let def = match state.get_definition(ptr.as_str()) {
                    Some(def) => def,
                    None => return Err(ValidationError::UndefinedDefinition),
                };
                let mut path = path.clone();
                path.push(i.to_string());
                if let Ok(_) = def.validate(state, value, path) {
                    found = true;
                    break;
                }
            }
            if !found {
                return Err(ValidationError::Failure {
                    rule: "items".to_string(),
                    path: path,
                    message: "Value must be a valid array item.".to_string(),
                });
            }
        }

        Ok(())
    }
}
